"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabPackagesDatasource = void 0;
const tslib_1 = require("tslib");
const decorator_1 = require("../../../util/cache/package/decorator");
const gitlab_1 = require("../../../util/http/gitlab");
const timestamp_1 = require("../../../util/timestamp");
const url_1 = require("../../../util/url");
const datasource_1 = require("../datasource");
const common_1 = require("./common");
// Gitlab Packages API: https://docs.gitlab.com/ee/api/packages.html
class GitlabPackagesDatasource extends datasource_1.Datasource {
    static id = common_1.datasource;
    http;
    caching = true;
    customRegistrySupport = true;
    defaultRegistryUrls = ['https://gitlab.com'];
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `created_at` field in the results.';
    constructor() {
        super(common_1.datasource);
        this.http = new gitlab_1.GitlabHttp(common_1.datasource);
    }
    static getGitlabPackageApiUrl(registryUrl, projectName, packageName) {
        const projectNameEncoded = encodeURIComponent(projectName);
        const packageNameEncoded = encodeURIComponent(packageName);
        return (0, url_1.joinUrlParts)(registryUrl, `api/v4/projects`, projectNameEncoded, `packages?package_name=${packageNameEncoded}&per_page=100`);
    }
    async getReleases({ registryUrl, packageName, }) {
        /* v8 ignore next 3 -- should never happen */
        if (!registryUrl) {
            return null;
        }
        const [projectPart, packagePart] = packageName.split(':', 2);
        const apiUrl = GitlabPackagesDatasource.getGitlabPackageApiUrl(registryUrl, projectPart, packagePart);
        const result = {
            releases: [],
        };
        let response;
        try {
            response = (await this.http.getJsonUnchecked(apiUrl, {
                paginate: true,
            })).body;
            result.releases = response
                // Setting the package_name option when calling the GitLab API isn't enough to filter information about other packages
                // because this option is only implemented on GitLab > 12.9 and it only does a fuzzy search.
                .filter((r) => (r.conan_package_name ?? r.name) === packagePart)
                .map(({ version, created_at }) => ({
                version,
                releaseTimestamp: (0, timestamp_1.asTimestamp)(created_at),
            }));
        }
        catch (err) {
            this.handleGenericErrors(err);
        }
        return result.releases?.length ? result : null;
    }
}
exports.GitlabPackagesDatasource = GitlabPackagesDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${common_1.datasource}`,
        key: ({ registryUrl, packageName }) => 
        // TODO: types (#22198)
        `${registryUrl}-${packageName}`,
    })
], GitlabPackagesDatasource.prototype, "getReleases", null);
//# sourceMappingURL=index.js.map