"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubTagsDatasource = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const graphql_1 = require("../../../util/github/graphql");
const tags_1 = require("../../../util/github/tags");
const url_1 = require("../../../util/github/url");
const memory_http_cache_provider_1 = require("../../../util/http/cache/memory-http-cache-provider");
const github_1 = require("../../../util/http/github");
const datasource_1 = require("../datasource");
class GithubTagsDatasource extends datasource_1.Datasource {
    static id = 'github-tags';
    defaultRegistryUrls = ['https://github.com'];
    registryStrategy = 'hunt';
    releaseTimestampSupport = true;
    // Note: not sure
    releaseTimestampNote = 'The get release timestamp is determined from the `releaseTimestamp` field in the results.';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'The source URL is determined by using the `packageName` and `registryUrl`.';
    http;
    constructor() {
        super(GithubTagsDatasource.id);
        this.http = new github_1.GithubHttp(GithubTagsDatasource.id);
    }
    async getCommit(registryUrl, githubRepo) {
        const apiBaseUrl = (0, url_1.getApiBaseUrl)(registryUrl);
        let digest = null;
        try {
            const url = `${apiBaseUrl}repos/${githubRepo}/commits?per_page=1`;
            const res = await this.http.getJsonUnchecked(url, {
                cacheProvider: memory_http_cache_provider_1.memCacheProvider,
            });
            digest = res.body[0].sha;
        }
        catch (err) {
            logger_1.logger.debug({ githubRepo, err, registryUrl }, 'Error getting latest commit from GitHub repo');
        }
        return digest;
    }
    /**
     * github.getDigest
     *
     * The `newValue` supplied here should be a valid tag for the docker image.
     *
     * Returns the latest commit hash for the repository.
     */
    getDigest({ packageName: repo, registryUrl }, newValue) {
        return newValue
            ? (0, tags_1.findCommitOfTag)(registryUrl, repo, newValue, this.http)
            : this.getCommit(registryUrl, repo);
    }
    async getReleases(config) {
        const { registryUrl, packageName: repo } = config;
        const sourceUrl = (0, url_1.getSourceUrl)(repo, registryUrl);
        const tagsResult = await (0, graphql_1.queryTags)(config, this.http);
        const releases = tagsResult.map(({ version, releaseTimestamp, gitRef, hash }) => ({
            newDigest: hash,
            version,
            releaseTimestamp,
            gitRef,
        }));
        try {
            // Fetch additional data from releases endpoint when possible
            const releasesResult = await (0, graphql_1.queryReleases)(config, this.http);
            const releasesMap = new Map();
            for (const release of releasesResult) {
                releasesMap.set(release.version, release);
            }
            for (const release of releases) {
                const isReleaseStable = releasesMap.get(release.version)?.isStable;
                if (is_1.default.boolean(isReleaseStable)) {
                    release.isStable = isReleaseStable;
                }
            }
        }
        catch (err) /* istanbul ignore next */ {
            logger_1.logger.debug({ err }, `Error fetching additional info for GitHub tags`);
        }
        const dependency = {
            sourceUrl,
            releases,
        };
        return dependency;
    }
}
exports.GithubTagsDatasource = GithubTagsDatasource;
//# sourceMappingURL=index.js.map