"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubReleasesDatasource = exports.cacheNamespace = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const graphql_1 = require("../../../util/github/graphql");
const tags_1 = require("../../../util/github/tags");
const url_1 = require("../../../util/github/url");
const github_1 = require("../../../util/http/github");
const datasource_1 = require("../datasource");
exports.cacheNamespace = 'datasource-github-releases';
class GithubReleasesDatasource extends datasource_1.Datasource {
    static id = 'github-releases';
    defaultRegistryUrls = ['https://github.com'];
    http;
    releaseTimestampSupport = true;
    // Note: not sure
    releaseTimestampNote = 'The release timestamp is determined from the `releaseTimestamp` field from the response.';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'The source URL is determined by using the `packageName` and `registryUrl`.';
    constructor() {
        super(GithubReleasesDatasource.id);
        this.http = new github_1.GithubHttp(GithubReleasesDatasource.id);
    }
    /**
     * Attempts to resolve the digest for the specified package.
     *
     * The `newValue` supplied here should be a valid tag for the GitHub release. The digest
     * of a GitHub release will be the underlying SHA of the release tag.
     *
     * Some managers like Bazel will deal with individual artifacts from releases and handle
     * the artifact checksum computation separately. This data-source does not know about
     * specific artifacts being used, as that could vary per manager
     */
    getDigest({ packageName: repo, currentValue, currentDigest, registryUrl, }, newValue) {
        logger_1.logger.debug({ repo, currentValue, currentDigest, registryUrl, newValue }, 'getDigest');
        return (0, tags_1.findCommitOfTag)(registryUrl, repo, newValue, this.http);
    }
    /**
     * This function can be used to fetch releases with a customizable versioning
     * (e.g. semver) and with releases.
     *
     * This function will:
     *  - Fetch all releases
     *  - Sanitize the versions if desired (e.g. strip out leading 'v')
     *  - Return a dependency object containing sourceUrl string and releases array
     */
    async getReleases(config) {
        const releasesResult = await (0, graphql_1.queryReleases)(config, this.http);
        const releases = releasesResult.map((item) => {
            const { version, releaseTimestamp, isStable } = item;
            const result = {
                version,
                gitRef: version,
                releaseTimestamp,
            };
            if (is_1.default.boolean(isStable)) {
                result.isStable = isStable;
            }
            return result;
        });
        const sourceUrl = (0, url_1.getSourceUrl)(config.packageName, config.registryUrl);
        return { sourceUrl, releases };
    }
}
exports.GithubReleasesDatasource = GithubReleasesDatasource;
//# sourceMappingURL=index.js.map