"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GiteaReleasesDatasource = void 0;
const tslib_1 = require("tslib");
const decorator_1 = require("../../../util/cache/package/decorator");
const gitea_1 = require("../../../util/http/gitea");
const datasource_1 = require("../datasource");
const gitea_tags_1 = require("../gitea-tags");
const schema_1 = require("../gitea-tags/schema");
const schema_2 = require("./schema");
class GiteaReleasesDatasource extends datasource_1.Datasource {
    static id = 'gitea-releases';
    http = new gitea_1.GiteaHttp(GiteaReleasesDatasource.id);
    static defaultRegistryUrls = ['https://gitea.com'];
    static cacheNamespace = `datasource-${GiteaReleasesDatasource.id}`;
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `published_at` field in the results.';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'The source URL is determined by using the `packageName` and `registryUrl`.';
    constructor() {
        super(GiteaReleasesDatasource.id);
    }
    // getReleases fetches list of tags for the repository
    async getReleases({ registryUrl, packageName: repo, }) {
        const url = `${gitea_tags_1.GiteaTagsDatasource.getApiUrl(registryUrl)}repos/${repo}/releases?draft=false`;
        const tags = (await this.http.getJson(url, {
            paginate: true,
        }, schema_2.ReleasesSchema)).body;
        const dependency = {
            sourceUrl: gitea_tags_1.GiteaTagsDatasource.getSourceUrl(repo, registryUrl),
            registryUrl: gitea_tags_1.GiteaTagsDatasource.getRegistryURL(registryUrl),
            releases: tags.map(({ tag_name, published_at, prerelease }) => ({
                version: tag_name,
                gitRef: tag_name,
                releaseTimestamp: published_at,
                isStable: !prerelease,
            })),
        };
        return dependency;
    }
    // getTagCommit fetched the commit has for specified tag
    async getTagCommit(registryUrl, repo, tag) {
        const url = `${gitea_tags_1.GiteaTagsDatasource.getApiUrl(registryUrl)}repos/${repo}/tags/${tag}`;
        const { body } = await this.http.getJson(url, schema_1.TagSchema);
        return body.commit.sha;
    }
    // getDigest fetched the latest commit for repository main branch
    // however, if newValue is provided, then getTagCommit is called
    async getDigest({ packageName: repo, registryUrl }, newValue) {
        if (newValue?.length) {
            return this.getTagCommit(registryUrl, repo, newValue);
        }
        const url = `${gitea_tags_1.GiteaTagsDatasource.getApiUrl(registryUrl)}repos/${repo}/commits?stat=false&verification=false&files=false&page=1&limit=1`;
        const { body } = await this.http.getJson(url, schema_1.CommitsSchema);
        if (body.length === 0) {
            return null;
        }
        return body[0].sha;
    }
}
exports.GiteaReleasesDatasource = GiteaReleasesDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: GiteaReleasesDatasource.cacheNamespace,
        key: ({ registryUrl, packageName }) => gitea_tags_1.GiteaTagsDatasource.getCacheKey(registryUrl, packageName, 'releases'),
    })
], GiteaReleasesDatasource.prototype, "getReleases", null);
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: GiteaReleasesDatasource.cacheNamespace,
        key: (registryUrl, repo, tag) => gitea_tags_1.GiteaTagsDatasource.getCacheKey(registryUrl, repo, `tag-${tag}`),
    })
], GiteaReleasesDatasource.prototype, "getTagCommit", null);
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: GiteaReleasesDatasource.cacheNamespace,
        key: ({ registryUrl, packageName }) => gitea_tags_1.GiteaTagsDatasource.getCacheKey(registryUrl, packageName, 'digest'),
    })
], GiteaReleasesDatasource.prototype, "getDigest", null);
//# sourceMappingURL=index.js.map