"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ecrPublicRegex = exports.ecrRegex = void 0;
exports.getECRAuthToken = getECRAuthToken;
exports.isECRMaxResultsError = isECRMaxResultsError;
const client_ecr_1 = require("@aws-sdk/client-ecr");
const logger_1 = require("../../../logger");
const regex_1 = require("../../../util/regex");
const sanitize_1 = require("../../../util/sanitize");
exports.ecrRegex = (0, regex_1.regEx)(/\d+\.(?:dkr\.ecr|dkr-ecr)(?:-fips)?\.([-a-z0-9]+)\.(?:amazonaws\.com|on\.aws)/);
exports.ecrPublicRegex = (0, regex_1.regEx)(/public\.ecr\.aws|ecr-public\.aws\.com/);
async function getECRAuthToken(region, opts) {
    const config = { region };
    if (opts.username === `AWS` && opts.password) {
        logger_1.logger.trace(`AWS user specified, encoding basic auth credentials for ECR registry`);
        return Buffer.from(`AWS:${opts.password}`).toString('base64');
    }
    else if (opts.username && opts.password) {
        logger_1.logger.trace(`Using AWS accessKey to get Authorization token for ECR registry`);
        config.credentials = {
            accessKeyId: opts.username,
            secretAccessKey: opts.password,
            ...(opts.token && { sessionToken: opts.token }),
        };
    }
    const ecr = new client_ecr_1.ECR(config);
    try {
        const data = await ecr.getAuthorizationToken({});
        const authorizationToken = data?.authorizationData?.[0]?.authorizationToken;
        if (authorizationToken) {
            // sanitize token
            (0, sanitize_1.addSecretForSanitizing)(authorizationToken);
            return authorizationToken;
        }
        logger_1.logger.warn('Could not extract authorizationToken from ECR getAuthorizationToken response');
    }
    catch (err) {
        logger_1.logger.trace({ err }, 'err');
        logger_1.logger.warn('ECR getAuthorizationToken error');
    }
    return null;
}
function isECRMaxResultsError(err) {
    const resp = err.response;
    return !!(resp?.statusCode === 405 &&
        resp.headers?.['docker-distribution-api-version'] &&
        // https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DescribeRepositories.html#ECR-DescribeRepositories-request-maxResults
        resp.body?.errors?.[0]?.message?.includes('Member must have value less than or equal to 1000'));
}
//# sourceMappingURL=ecr.js.map