"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerHubCache = void 0;
const tslib_1 = require("tslib");
const dequal_1 = require("dequal");
const luxon_1 = require("luxon");
const packageCache = tslib_1.__importStar(require("../../../util/cache/package"));
const cacheNamespace = 'datasource-docker-hub-cache';
class DockerHubCache {
    dockerRepository;
    cache;
    isChanged = false;
    reconciledIds = new Set();
    constructor(dockerRepository, cache) {
        this.dockerRepository = dockerRepository;
        this.cache = cache;
    }
    static async init(dockerRepository) {
        let repoCache = await packageCache.get(cacheNamespace, dockerRepository);
        repoCache ??= {
            items: {},
            updatedAt: null,
        };
        return new DockerHubCache(dockerRepository, repoCache);
    }
    reconcile(items, expectedCount) {
        let needNextPage = true;
        let earliestDate = null;
        let { updatedAt } = this.cache;
        let latestDate = updatedAt ? luxon_1.DateTime.fromISO(updatedAt) : null;
        for (const newItem of items) {
            const id = newItem.id;
            this.reconciledIds.add(id);
            const oldItem = this.cache.items[id];
            const itemDate = luxon_1.DateTime.fromISO(newItem.last_updated);
            if (!earliestDate || earliestDate > itemDate) {
                earliestDate = itemDate;
            }
            if (!latestDate || latestDate < itemDate) {
                latestDate = itemDate;
                updatedAt = newItem.last_updated;
            }
            if ((0, dequal_1.dequal)(oldItem, newItem)) {
                needNextPage = false;
                continue;
            }
            this.cache.items[newItem.id] = newItem;
            this.isChanged = true;
        }
        this.cache.updatedAt = updatedAt;
        if (earliestDate && latestDate) {
            for (const [key, item] of Object.entries(this.cache.items)) {
                const id = parseInt(key, 10);
                const itemDate = luxon_1.DateTime.fromISO(item.last_updated);
                if (itemDate < earliestDate ||
                    itemDate > latestDate ||
                    this.reconciledIds.has(id)) {
                    continue;
                }
                delete this.cache.items[id];
                this.isChanged = true;
            }
            if (Object.keys(this.cache.items).length > expectedCount) {
                return true;
            }
        }
        return needNextPage;
    }
    async save() {
        if (this.isChanged) {
            await packageCache.set(cacheNamespace, this.dockerRepository, this.cache, 3 * 60 * 24 * 30);
        }
    }
    getItems() {
        return Object.values(this.cache.items);
    }
}
exports.DockerHubCache = DockerHubCache;
//# sourceMappingURL=dockerhub-cache.js.map