import { Datasource } from '../datasource';
import type { GetReleasesConfig, ReleaseResult } from '../types';
import type { PackageDescription } from './types';
export declare class DebDatasource extends Datasource {
    static readonly id = "deb";
    constructor();
    /**
     * Users are able to specify custom Debian repositories as long as they follow
     * the Debian package repository format as specified here
     * @see{https://wiki.debian.org/DebianRepository/Format}
     */
    readonly customRegistrySupport = true;
    /**
     * Users can specify multiple upstream repositories and the datasource will aggregate the release
     * @example
     * When specifying multiple dependencies both internal and external dependencies from internal/external artifactory
     */
    readonly registryStrategy = "merge";
    /**
     * The original apt source list file format is
     * deb uri distribution [component1] [component2] [...]
     * @see{https://wiki.debian.org/DebianRepository/Format}
     *
     * However, for Renovate, we require the registry URLs to be
     * valid URLs which is why the parameters are encoded in the URL.
     *
     * The following query parameters are required:
     * - components: comma separated list of components
     * - suite: stable, oldstable or other alias for a release, either this or release must be given like buster
     * - binaryArch: e.g. amd64 resolves to http://deb.debian.org/debian/dists/stable/non-free/binary-amd64/
     */
    readonly defaultRegistryUrls: string[];
    readonly defaultVersioning = "deb";
    /**
     * Downloads and extracts a package file from a component URL.
     *
     * @param componentUrl - The URL of the component.
     * @returns The path to the extracted file and the last modification timestamp.
     * @throws Will throw an error if no valid compression method is found.
     */
    private downloadAndExtractPackage;
    /**
     * Downloads a package file if it has been modified since the last download timestamp.
     *
     * @param basePackageUrl - The base URL of the package.
     * @param compression - The compression method used (e.g., 'gz').
     * @param compressedFile - The path where the compressed file will be saved.
     * @param lastDownloadTimestamp - The timestamp of the last download.
     * @returns True if the file was downloaded, otherwise false.
     */
    private downloadPackageFile;
    /**
     * Fetches the content of the InRelease file from the given base suite URL.
     *
     * @param baseReleaseUrl - The base URL of the suite (e.g., 'https://deb.debian.org/debian/dists/bullseye').
     * @returns resolves to the content of the InRelease file.
     * @throws An error if the InRelease file could not be downloaded.
     */
    private fetchInReleaseFile;
    /**
     * Checks if a packageUrl content has been modified since the specified timestamp.
     *
     * @param packageUrl - The URL to check.
     * @param lastDownloadTimestamp - The timestamp of the last download.
     * @returns True if the content has been modified, otherwise false.
     * @see https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/If-Modified-Since
     */
    private checkIfModified;
    /**
     * Parses the extracted package index file.
     *
     * @param extractedFile - The path to the extracted package file.
     * @param lastTimestamp - The timestamp of the last modification.
     * @returns a list of packages with minimal Metadata.
     */
    parseExtractedPackageIndex(extractedFile: string, _lastTimestamp: Date): Promise<Record<string, PackageDescription[]>>;
    getPackageIndex(componentUrl: string): Promise<Record<string, PackageDescription[]>>;
    /**
     * Fetches the release information for a given package from the registry URL.
     *
     * @param config - Configuration for fetching releases.
     * @returns The release result if the package is found, otherwise null.
     */
    getReleases({ registryUrl, packageName, }: GetReleasesConfig): Promise<ReleaseResult | null>;
}
