"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DartVersionDatasource = exports.svnVersionRegex = exports.stableVersionRegex = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const regex_1 = require("../../../util/regex");
const datasource_1 = require("../datasource");
exports.stableVersionRegex = (0, regex_1.regEx)(/^\d+\.\d+\.\d+$/);
/**
 * The server returns old svn versions which would need mapping to a version.
 * They are very old, so we skip them instead.
 * https://github.com/dart-lang/site-www/blob/7d4409c87bb6570b2d8870b20283f81f2b7e08fc/tool/get-dart/dart_sdk_archive/lib/src/svn_versions.dart#L2
 */
exports.svnVersionRegex = (0, regex_1.regEx)(/^\d+$/);
class DartVersionDatasource extends datasource_1.Datasource {
    static id = 'dart-version';
    constructor() {
        super(DartVersionDatasource.id);
    }
    customRegistrySupport = false;
    defaultRegistryUrls = ['https://storage.googleapis.com'];
    caching = true;
    channels = ['stable', 'beta', 'dev'];
    sourceUrlSupport = 'package';
    sourceUrlNote = 'We use the URL: https://github.com/dart-lang/sdk.';
    async getReleases({ registryUrl, }) {
        /* v8 ignore next 3 -- should never happen */
        if (!registryUrl) {
            return null;
        }
        const result = {
            homepage: 'https://dart.dev/',
            sourceUrl: 'https://github.com/dart-lang/sdk',
            registryUrl,
            releases: [],
        };
        try {
            for (const channel of this.channels) {
                const resp = (await this.http.getJsonUnchecked(`${registryUrl}/storage/v1/b/dart-archive/o?delimiter=%2F&prefix=channels%2F${channel}%2Frelease%2F&alt=json`)).body;
                const releases = this.getReleasesFromResponse(channel, resp.prefixes);
                result.releases.push(...releases);
            }
        }
        catch (err) {
            this.handleGenericErrors(err);
        }
        return result.releases.length ? result : null;
    }
    getReleasesFromResponse(channel, prefixes) {
        return prefixes
            .map((prefix) => this.getVersionFromPrefix(prefix))
            .filter(is_1.default.string)
            .filter((version) => {
            if (version === 'latest' ||
                // skip old svn versions
                exports.svnVersionRegex.test(version) ||
                // The API response contains a stable version being released as a non-stable
                // release. So we filter out these releases here.
                (channel !== 'stable' && exports.stableVersionRegex.test(version))) {
                return false;
            }
            return true;
        })
            .map((version) => ({ version, isStable: channel === 'stable' }));
    }
    // Prefix should have a format of "channels/stable/release/2.9.3/"
    getVersionFromPrefix(prefix) {
        const parts = prefix.split('/');
        return parts[parts.length - 2];
    }
}
exports.DartVersionDatasource = DartVersionDatasource;
//# sourceMappingURL=index.js.map