"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomDatasource = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const jsonata_1 = require("../../../util/jsonata");
const datasource_1 = require("../datasource");
const formats_1 = require("./formats");
const schema_1 = require("./schema");
const utils_1 = require("./utils");
class CustomDatasource extends datasource_1.Datasource {
    static id = 'custom';
    customRegistrySupport = true;
    constructor() {
        super(CustomDatasource.id);
    }
    async getReleases(getReleasesConfig) {
        const config = (0, utils_1.getCustomConfig)(getReleasesConfig);
        if (is_1.default.nullOrUndefined(config)) {
            return null;
        }
        const { defaultRegistryUrlTemplate, transformTemplates, format } = config;
        const fetcher = formats_1.fetchers[format];
        const isLocalRegistry = defaultRegistryUrlTemplate.startsWith('file://');
        let data;
        try {
            if (isLocalRegistry) {
                data = await fetcher.readFile(defaultRegistryUrlTemplate.replace('file://', ''));
            }
            else {
                data = await fetcher.fetch(this.http, defaultRegistryUrlTemplate);
            }
        }
        catch (e) {
            this.handleHttpErrors(e);
            return null;
        }
        logger_1.logger.trace({ data }, `Custom datasource API fetcher '${format}' received data. Starting transformation.`);
        for (const transformTemplate of transformTemplates) {
            const expression = (0, jsonata_1.getExpression)(transformTemplate);
            if (expression instanceof Error) {
                logger_1.logger.once.warn({ errorMessage: expression.message }, `Invalid JSONata expression: ${transformTemplate}`);
                return null;
            }
            try {
                const modifiedData = await expression.evaluate(data);
                logger_1.logger.trace({ before: data, after: modifiedData }, `Custom datasource transformed data.`);
                data = modifiedData;
            }
            catch (err) {
                logger_1.logger.once.warn({ err }, `Error while evaluating JSONata expression: ${transformTemplate}`);
                return null;
            }
        }
        try {
            const parsed = schema_1.ReleaseResultZodSchema.parse(data);
            return structuredClone(parsed);
        }
        catch (err) {
            logger_1.logger.debug({ err }, `Response has failed validation`);
            logger_1.logger.trace({ data }, 'Response that has failed validation');
            return null;
        }
    }
    getDigest({ packageName }, newValue) {
        // Return null here to support setting a digest: value can be provided digest in getReleases
        return Promise.resolve(null);
    }
}
exports.CustomDatasource = CustomDatasource;
//# sourceMappingURL=index.js.map