"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CondaDatasource = void 0;
const tslib_1 = require("tslib");
const logger_1 = require("../../../logger");
const external_host_error_1 = require("../../../types/errors/external-host-error");
const array_1 = require("../../../util/array");
const decorator_1 = require("../../../util/cache/package/decorator");
const http_1 = require("../../../util/http");
const timestamp_1 = require("../../../util/timestamp");
const url_1 = require("../../../util/url");
const datasource_1 = require("../datasource");
const common_1 = require("./common");
const prefixDev = tslib_1.__importStar(require("./prefix-dev"));
class CondaDatasource extends datasource_1.Datasource {
    static id = common_1.datasource;
    constructor() {
        super(common_1.datasource);
    }
    customRegistrySupport = true;
    registryStrategy = 'hunt';
    defaultRegistryUrls = [common_1.defaultRegistryUrl];
    caching = true;
    sourceUrlSupport = 'package';
    sourceUrlNote = 'The source URL is determined from the `dev_url` field in the results.';
    async getReleases({ registryUrl, packageName, }) {
        logger_1.logger.trace({ registryUrl, packageName }, 'fetching conda package');
        if (!registryUrl) {
            return null;
        }
        // fast.prefix.dev is a alias, deprecated, but still running.
        // We expect registryUrl to be `https://prefix.dev/${channel}` here.
        if (registryUrl.startsWith('https://prefix.dev/') ||
            registryUrl.startsWith('https://fast.prefix.dev/')) {
            // Since the registryUrl contains at least 3 `/` ,
            // the channel varitable won't be undefined in any case.
            const channel = (0, url_1.ensureTrailingSlash)(registryUrl).split('/').at(-2);
            return prefixDev.getReleases(this.http, channel, packageName);
        }
        const url = (0, url_1.joinUrlParts)(registryUrl, packageName);
        const result = {
            releases: [],
        };
        let response;
        try {
            response = await this.http.getJsonUnchecked(url);
            result.homepage = response.body.html_url;
            result.sourceUrl = response.body.dev_url;
            const releaseDate = {};
            // we assume all packages are roughly released on the same time
            for (const file of (0, array_1.coerceArray)(response.body.files)) {
                releaseDate[file.version] ??= timestamp_1.Timestamp.parse(file.upload_time);
            }
            response.body.versions.forEach((version) => {
                const thisRelease = {
                    version,
                    releaseTimestamp: releaseDate[version],
                };
                result.releases.push(thisRelease);
            });
        }
        catch (err) {
            if (err instanceof http_1.HttpError) {
                if (err.response?.statusCode !== 404) {
                    throw new external_host_error_1.ExternalHostError(err);
                }
            }
            this.handleGenericErrors(err);
        }
        return result.releases.length ? result : null;
    }
}
exports.CondaDatasource = CondaDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${common_1.datasource}`,
        key: ({ registryUrl, packageName }) => 
        // TODO: types (#22198)
        `${registryUrl}:${packageName}`,
    })
], CondaDatasource.prototype, "getReleases", null);
//# sourceMappingURL=index.js.map