"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildpacksRegistryDatasource = void 0;
const tslib_1 = require("tslib");
const url_join_1 = tslib_1.__importDefault(require("url-join"));
const zod_1 = require("zod");
const logger_1 = require("../../../logger");
const decorator_1 = require("../../../util/cache/package/decorator");
const result_1 = require("../../../util/result");
const datasource_1 = require("../datasource");
const schema_1 = require("../schema");
const schema_2 = require("./schema");
class BuildpacksRegistryDatasource extends datasource_1.Datasource {
    static id = 'buildpacks-registry';
    constructor() {
        super(BuildpacksRegistryDatasource.id);
    }
    customRegistrySupport = false;
    defaultRegistryUrls = ['https://registry.buildpacks.io'];
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `published_at` field in the results.';
    sourceUrlSupport = 'release';
    sourceUrlNote = 'The source URL is determined from the `source_code_url` field of the release object in the results.';
    async getReleases(config) {
        const result = result_1.Result.parse(config, schema_1.ReleasesConfig)
            .transform(({ packageName, registryUrl }) => {
            const url = (0, url_join_1.default)(registryUrl, 'api', 'v1', 'buildpacks', packageName);
            return this.http.getJsonSafe(url, schema_2.BuildpacksRegistryResponseSchema);
        })
            .transform(({ versions, latest }) => {
            const releases = versions;
            const res = { releases };
            if (latest?.homepage) {
                res.homepage = latest.homepage;
            }
            return res;
        });
        const { val, err } = await result.unwrap();
        if (err instanceof zod_1.ZodError) {
            logger_1.logger.debug({ err }, 'buildpacks: validation error');
            return null;
        }
        if (err) {
            this.handleGenericErrors(err);
        }
        return val;
    }
}
exports.BuildpacksRegistryDatasource = BuildpacksRegistryDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${BuildpacksRegistryDatasource.id}`,
        key: ({ registryUrl, packageName }) => `${registryUrl}:${packageName}`,
    })
], BuildpacksRegistryDatasource.prototype, "getReleases", null);
//# sourceMappingURL=index.js.map