"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BitbucketServerTagsDatasource = void 0;
const tslib_1 = require("tslib");
const zod_1 = require("zod");
const logger_1 = require("../../../logger");
const decorator_1 = require("../../../util/cache/package/decorator");
const bitbucket_server_1 = require("../../../util/http/bitbucket-server");
const regex_1 = require("../../../util/regex");
const result_1 = require("../../../util/result");
const url_1 = require("../../../util/url");
const datasource_1 = require("../datasource");
const schema_1 = require("../schema");
const schema_2 = require("./schema");
class BitbucketServerTagsDatasource extends datasource_1.Datasource {
    static id = 'bitbucket-server-tags';
    http = new bitbucket_server_1.BitbucketServerHttp(BitbucketServerTagsDatasource.id);
    static sourceUrlSupport = 'package';
    static sourceUrlNote = 'The source URL is determined by using the `packageName` and `registryUrl`.';
    static cacheNamespace = `datasource-${BitbucketServerTagsDatasource.id}`;
    constructor() {
        super(BitbucketServerTagsDatasource.id);
    }
    static getRegistryURL(registryUrl) {
        return registryUrl?.replace((0, regex_1.regEx)(/\/rest\/api\/1.0$/), '');
    }
    static getSourceUrl(projectKey, repositorySlug, registryUrl) {
        const url = BitbucketServerTagsDatasource.getRegistryURL(registryUrl);
        return `${(0, url_1.ensureTrailingSlash)(url)}projects/${projectKey}/repos/${repositorySlug}`;
    }
    static getApiUrl(registryUrl) {
        const res = BitbucketServerTagsDatasource.getRegistryURL(registryUrl);
        return `${(0, url_1.ensureTrailingSlash)(res)}rest/api/1.0/`;
    }
    static getCacheKey(registryUrl, repo, type) {
        return `${BitbucketServerTagsDatasource.getRegistryURL(registryUrl ?? '')}:${repo}:${type}`;
    }
    // getReleases fetches list of tags for the repository
    async getReleases(config) {
        const { registryUrl, packageName } = config;
        const [projectKey, repositorySlug] = packageName.split('/');
        if (!registryUrl) {
            logger_1.logger.debug('Missing registryUrl');
            return null;
        }
        const result = result_1.Result.parse(config, schema_1.ReleasesConfig)
            .transform(({ registryUrl }) => {
            const url = `${BitbucketServerTagsDatasource.getApiUrl(registryUrl)}projects/${projectKey}/repos/${repositorySlug}/tags`;
            return this.http.getJsonSafe(url, { paginate: true }, schema_2.BitbucketServerTags);
        })
            .transform((tags) => tags.map(({ displayId, hash }) => ({
            version: displayId,
            gitRef: displayId,
            newDigest: hash ?? undefined,
        })))
            .transform((versions) => {
            return {
                sourceUrl: BitbucketServerTagsDatasource.getSourceUrl(projectKey, repositorySlug, registryUrl),
                registryUrl: BitbucketServerTagsDatasource.getRegistryURL(registryUrl),
                releases: versions,
            };
        });
        const { val, err } = await result.unwrap();
        if (err instanceof zod_1.ZodError) {
            logger_1.logger.debug({ err }, 'bitbucket-server-tags: validation error');
            return null;
        }
        if (err) {
            this.handleGenericErrors(err);
        }
        return val;
    }
    // getTagCommit fetches the commit hash for the specified tag
    async getTagCommit(baseUrl, tag) {
        const bitbucketServerTag = (await this.http.getJson(`${baseUrl}/tags/${tag}`, schema_2.BitbucketServerTag)).body;
        return bitbucketServerTag.hash ?? null;
    }
    // getDigest fetches the latest commit for repository main branch.
    // If newValue is provided, then getTagCommit is called
    async getDigest(config, newValue) {
        const { registryUrl, packageName } = config;
        const [projectKey, repositorySlug] = packageName.split('/');
        if (!registryUrl) {
            logger_1.logger.debug('Missing registryUrl');
            return null;
        }
        const baseUrl = `${BitbucketServerTagsDatasource.getApiUrl(registryUrl)}projects/${projectKey}/repos/${repositorySlug}`;
        if (newValue?.length) {
            return this.getTagCommit(baseUrl, newValue);
        }
        const result = result_1.Result.parse(config, schema_1.DigestsConfig)
            .transform(() => {
            const url = `${baseUrl}/commits?ignoreMissing=true`;
            return this.http.getJsonSafe(url, {
                paginate: true,
                limit: 1,
                maxPages: 1,
            }, schema_2.BitbucketServerCommits);
        })
            .transform((commits) => {
            return commits[0]?.id;
        });
        const { val = null, err } = await result.unwrap();
        if (err instanceof zod_1.ZodError) {
            logger_1.logger.debug({ err }, 'bitbucket-server-tags: validation error');
            return null;
        }
        if (err) {
            this.handleGenericErrors(err);
        }
        return val;
    }
}
exports.BitbucketServerTagsDatasource = BitbucketServerTagsDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: BitbucketServerTagsDatasource.cacheNamespace,
        key: ({ registryUrl, packageName }) => BitbucketServerTagsDatasource.getCacheKey(registryUrl, packageName, 'tags'),
    })
], BitbucketServerTagsDatasource.prototype, "getReleases", null);
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: BitbucketServerTagsDatasource.cacheNamespace,
        key: ({ registryUrl, packageName }, tag) => BitbucketServerTagsDatasource.getCacheKey(registryUrl, packageName, `tag-${tag}`),
    })
], BitbucketServerTagsDatasource.prototype, "getTagCommit", null);
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: BitbucketServerTagsDatasource.cacheNamespace,
        key: ({ registryUrl, packageName }) => BitbucketServerTagsDatasource.getCacheKey(registryUrl, packageName, 'digest'),
    })
], BitbucketServerTagsDatasource.prototype, "getDigest", null);
//# sourceMappingURL=index.js.map