"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BazelDatasource = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const external_host_error_1 = require("../../../types/errors/external-host-error");
const decorator_1 = require("../../../util/cache/package/decorator");
const fs_1 = require("../../../util/fs");
const http_1 = require("../../../util/http");
const url_1 = require("../../../util/url");
const bazel_module_1 = require("../../versioning/bazel-module");
const bzlmod_version_1 = require("../../versioning/bazel-module/bzlmod-version");
const datasource_1 = require("../datasource");
const schema_1 = require("./schema");
class BazelDatasource extends datasource_1.Datasource {
    static id = 'bazel';
    static bazelCentralRepoUrl = 'https://raw.githubusercontent.com/bazelbuild/bazel-central-registry/main';
    defaultRegistryUrls = [BazelDatasource.bazelCentralRepoUrl];
    registryStrategy = 'hunt';
    customRegistrySupport = true;
    caching = true;
    defaultVersioning = bazel_module_1.id;
    static packageMetadataPath(packageName) {
        return `/modules/${packageName}/metadata.json`;
    }
    constructor() {
        super(BazelDatasource.id);
    }
    async getReleases({ registryUrl, packageName, }) {
        const path = BazelDatasource.packageMetadataPath(packageName);
        const url = (0, url_1.joinUrlParts)(registryUrl, path);
        const result = { releases: [] };
        try {
            let metadata;
            const FILE_PREFIX = 'file://';
            if (url.startsWith(FILE_PREFIX)) {
                const filePath = url.slice(FILE_PREFIX.length);
                if (!(0, fs_1.isValidLocalPath)(filePath)) {
                    return null;
                }
                const fileContent = await (0, fs_1.readLocalFile)(filePath, 'utf8');
                if (!fileContent) {
                    return null;
                }
                metadata = schema_1.BazelModuleMetadata.parse(JSON.parse(fileContent));
            }
            else {
                const response = await this.http.getJson(url, schema_1.BazelModuleMetadata);
                metadata = response.body;
            }
            result.releases = metadata.versions
                .map((v) => new bzlmod_version_1.BzlmodVersion(v))
                .sort(bzlmod_version_1.BzlmodVersion.defaultCompare)
                .map((bv) => {
                const release = { version: bv.original };
                if (is_1.default.truthy(metadata.yanked_versions?.[bv.original])) {
                    release.isDeprecated = true;
                }
                return release;
            });
            if (metadata.homepage) {
                result.homepage = metadata.homepage;
            }
        }
        catch (err) {
            if (err instanceof http_1.HttpError) {
                if (err.response?.statusCode === 404) {
                    return null;
                }
                throw new external_host_error_1.ExternalHostError(err);
            }
            this.handleGenericErrors(err);
        }
        return result.releases.length ? result : null;
    }
}
exports.BazelDatasource = BazelDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${BazelDatasource.id}`,
        key: ({ registryUrl, packageName }) => `${registryUrl}:${packageName}`,
    })
], BazelDatasource.prototype, "getReleases", null);
//# sourceMappingURL=index.js.map