"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resetReport = resetReport;
exports.addBranchStats = addBranchStats;
exports.addExtractionStats = addExtractionStats;
exports.addLibYears = addLibYears;
exports.finalizeReport = finalizeReport;
exports.exportStats = exportStats;
exports.getReport = getReport;
const tslib_1 = require("tslib");
const client_s3_1 = require("@aws-sdk/client-s3");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../logger");
const fs_1 = require("../util/fs");
const s3_1 = require("../util/s3");
const report = {
    problems: [],
    repositories: {},
};
/**
 * Reset the report
 * Should only be used for testing
 */
function resetReport() {
    report.problems = [];
    report.repositories = {};
}
function addBranchStats(config, branchesInformation) {
    if (is_1.default.nullOrUndefined(config.reportType)) {
        return;
    }
    coerceRepo(config.repository);
    report.repositories[config.repository].branches = branchesInformation;
}
function addExtractionStats(config, extractResult) {
    if (is_1.default.nullOrUndefined(config.reportType)) {
        return;
    }
    coerceRepo(config.repository);
    report.repositories[config.repository].packageFiles =
        extractResult.packageFiles;
}
function addLibYears(config, libYearsWithDepCount) {
    if (is_1.default.nullOrUndefined(config.reportType)) {
        return;
    }
    coerceRepo(config.repository);
    report.repositories[config.repository].libYearsWithStatus =
        libYearsWithDepCount;
}
function finalizeReport() {
    const allProblems = structuredClone((0, logger_1.getProblems)());
    for (const problem of allProblems) {
        const repository = problem.repository;
        delete problem.repository;
        // if the problem can be connected to a repository add it their else add to the root list
        if (repository) {
            coerceRepo(repository);
            report.repositories[repository].problems.push(problem);
        }
        else {
            report.problems.push(problem);
        }
    }
}
async function exportStats(config) {
    try {
        if (is_1.default.nullOrUndefined(config.reportType)) {
            return;
        }
        if (config.reportType === 'logging') {
            logger_1.logger.info({ report }, 'Printing report');
            return;
        }
        if (config.reportType === 'file') {
            const path = config.reportPath;
            await (0, fs_1.writeSystemFile)(path, JSON.stringify(report));
            logger_1.logger.debug({ path }, 'Writing report');
            return;
        }
        if (config.reportType === 's3') {
            const s3Url = (0, s3_1.parseS3Url)(config.reportPath);
            if (is_1.default.nullOrUndefined(s3Url)) {
                logger_1.logger.warn({ reportPath: config.reportPath }, 'Failed to parse s3 URL');
                return;
            }
            const s3Params = {
                Bucket: s3Url.Bucket,
                Key: s3Url.Key,
                Body: JSON.stringify(report),
                ContentType: 'application/json',
            };
            const client = (0, s3_1.getS3Client)(config.s3Endpoint, config.s3PathStyle);
            const command = new client_s3_1.PutObjectCommand(s3Params);
            await client.send(command);
        }
    }
    catch (err) {
        logger_1.logger.warn({ err }, 'Reporting.exportStats() - failure');
    }
}
function getReport() {
    return structuredClone(report);
}
function coerceRepo(repository) {
    if (!is_1.default.undefined(report.repositories[repository])) {
        return;
    }
    report.repositories[repository] = {
        problems: [],
        branches: [],
        packageFiles: {},
    };
}
//# sourceMappingURL=reporting.js.map