"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getParentName = getParentName;
exports.validatePlainObject = validatePlainObject;
exports.validateNumber = validateNumber;
exports.isFalseGlobal = isFalseGlobal;
exports.validateRegexManagerFields = validateRegexManagerFields;
exports.validateJSONataManagerFields = validateJSONataManagerFields;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const jsonata_1 = tslib_1.__importDefault(require("jsonata"));
const logger_1 = require("../../logger");
const regex_1 = require("../../util/regex");
function getParentName(parentPath) {
    return parentPath
        ? parentPath
            .replace((0, regex_1.regEx)(/\.?encrypted$/), '')
            .replace((0, regex_1.regEx)(/\[\d+\]$/), '')
            .split('.')
            .pop()
        : '.';
}
function validatePlainObject(val) {
    for (const [key, value] of Object.entries(val)) {
        if (!is_1.default.string(value)) {
            return key;
        }
    }
    return true;
}
function validateNumber(key, val, allowsNegative, currentPath, subKey) {
    const errors = [];
    const path = `${currentPath}${subKey ? '.' + subKey : ''}`;
    if (is_1.default.number(val)) {
        if (val < 0 && !allowsNegative) {
            errors.push({
                topic: 'Configuration Error',
                message: `Configuration option \`${path}\` should be a positive integer. Found negative value instead.`,
            });
        }
    }
    else {
        errors.push({
            topic: 'Configuration Error',
            message: `Configuration option \`${path}\` should be an integer. Found: ${JSON.stringify(val)} (${typeof val}).`,
        });
    }
    return errors;
}
/**  An option is a false global if it has the same name as a global only option
 *   but is actually just the field of a non global option or field an children of the non global option
 *   eg. token: it's global option used as the bot's token as well and
 *   also it can be the token used for a platform inside the hostRules configuration
 */
function isFalseGlobal(optionName, parentPath) {
    if (parentPath?.includes('hostRules')) {
        if (optionName === 'token' ||
            optionName === 'username' ||
            optionName === 'password') {
            return true;
        }
    }
    return false;
}
function hasField(customManager, field) {
    const templateField = `${field}Template`;
    const fieldStr = customManager.customType === 'regex' ? `(?<${field}>` : field;
    return !!(customManager[templateField] ??
        customManager.matchStrings?.some((matchString) => matchString.includes(fieldStr)));
}
function validateRegexManagerFields(customManager, currentPath, errors) {
    if (is_1.default.nonEmptyArray(customManager.matchStrings)) {
        for (const matchString of customManager.matchStrings) {
            try {
                (0, regex_1.regEx)(matchString);
            }
            catch (err) {
                logger_1.logger.debug({ err }, 'customManager.matchStrings regEx validation error');
                errors.push({
                    topic: 'Configuration Error',
                    message: `Invalid regExp for ${currentPath}: \`${matchString}\``,
                });
            }
        }
    }
    else {
        errors.push({
            topic: 'Configuration Error',
            message: 'Each Custom Manager `matchStrings` array must have at least one item.',
        });
    }
    const mandatoryFields = ['currentValue', 'datasource'];
    for (const field of mandatoryFields) {
        if (!hasField(customManager, field)) {
            errors.push({
                topic: 'Configuration Error',
                message: `Regex Managers must contain ${field}Template configuration or regex group named ${field}`,
            });
        }
    }
    const nameFields = ['depName', 'packageName'];
    if (!nameFields.some((field) => hasField(customManager, field))) {
        errors.push({
            topic: 'Configuration Error',
            message: `Regex Managers must contain depName or packageName regex groups or templates`,
        });
    }
}
function validateJSONataManagerFields(customManager, currentPath, errors) {
    if (!is_1.default.nonEmptyString(customManager.fileFormat)) {
        errors.push({
            topic: 'Configuration Error',
            message: 'Each JSONata manager must contain a fileFormat field.',
        });
    }
    if (is_1.default.nonEmptyArray(customManager.matchStrings)) {
        for (const matchString of customManager.matchStrings) {
            try {
                (0, jsonata_1.default)(matchString);
            }
            catch (err) {
                logger_1.logger.debug({ err }, 'customManager.matchStrings JSONata query validation error');
                errors.push({
                    topic: 'Configuration Error',
                    message: `Invalid JSONata query for ${currentPath}: \`${matchString}\``,
                });
            }
        }
    }
    else {
        errors.push({
            topic: 'Configuration Error',
            message: `Each Custom Manager must contain a non-empty matchStrings array`,
        });
    }
    const mandatoryFields = ['currentValue', 'datasource'];
    for (const field of mandatoryFields) {
        if (!hasField(customManager, field)) {
            errors.push({
                topic: 'Configuration Error',
                message: `JSONata Managers must contain ${field}Template configuration or ${field} in the query `,
            });
        }
    }
    const nameFields = ['depName', 'packageName'];
    if (!nameFields.some((field) => hasField(customManager, field))) {
        errors.push({
            topic: 'Configuration Error',
            message: `JSONata Managers must contain depName or packageName in the query or their templates`,
        });
    }
}
//# sourceMappingURL=utils.js.map