"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parsePreset = parsePreset;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const regex_1 = require("../../util/regex");
const url_1 = require("../../util/url");
const util_1 = require("./util");
const nonScopedPresetWithSubdirRegex = (0, regex_1.regEx)(/^(?<repo>~?[\w\-. /%]+?)\/\/(?:(?<presetPath>[\w\-./]+)\/)?(?<presetName>[\w\-.]+)(?:#(?<tag>[\w\-./]+?))?$/);
const gitPresetRegex = (0, regex_1.regEx)(/^(?<repo>~?[\w\-. /%]+)(?::(?<presetName>[\w\-.+/]+))?(?:#(?<tag>[\w\-./]+?))?$/);
function parsePreset(input) {
    let str = input;
    let presetSource;
    let presetPath;
    let repo;
    let presetName;
    let tag;
    let rawParams;
    let params;
    if (str.startsWith('github>')) {
        presetSource = 'github';
        str = str.substring('github>'.length);
    }
    else if (str.startsWith('gitlab>')) {
        presetSource = 'gitlab';
        str = str.substring('gitlab>'.length);
    }
    else if (str.startsWith('gitea>')) {
        presetSource = 'gitea';
        str = str.substring('gitea>'.length);
    }
    else if (str.startsWith('local>')) {
        presetSource = 'local';
        str = str.substring('local>'.length);
    }
    else if ((0, url_1.isHttpUrl)(str)) {
        presetSource = 'http';
    }
    else if (!str.startsWith('@') &&
        !str.startsWith(':') &&
        str.includes('/')) {
        presetSource = 'local';
    }
    str = str.replace((0, regex_1.regEx)(/^npm>/), '');
    presetSource = presetSource ?? 'npm';
    if (str.includes('(')) {
        rawParams = str.slice(str.indexOf('(') + 1, -1);
        params = rawParams.split(',').map((elem) => elem.trim());
        str = str.slice(0, str.indexOf('('));
    }
    if (presetSource === 'http') {
        return { presetSource, repo: str, presetName: '', params, rawParams };
    }
    const presetsPackages = [
        'abandonments',
        'compatibility',
        'config',
        'customManagers',
        'default',
        'docker',
        'global',
        'group',
        'helpers',
        'mergeConfidence',
        'monorepo',
        'npm',
        'packages',
        'preview',
        'replacements',
        'schedule',
        'security',
        'workarounds',
    ];
    if (presetsPackages.some((presetPackage) => str.startsWith(`${presetPackage}:`))) {
        presetSource = 'internal';
        [repo, presetName] = str.split(':');
    }
    else if (str.startsWith(':')) {
        // default namespace
        presetSource = 'internal';
        repo = 'default';
        presetName = str.slice(1);
    }
    else if (str.startsWith('@')) {
        // scoped namespace
        [, repo] = (0, regex_1.regEx)(/(@.*?)(:|$)/).exec(str);
        str = str.slice(repo.length);
        if (!repo.includes('/')) {
            repo += '/renovate-config';
        }
        if (str === '') {
            presetName = 'default';
        }
        else {
            presetName = str.slice(1);
        }
    }
    else if (str.includes('//')) {
        // non-scoped namespace with a subdirectory preset
        // Validation
        if (str.includes(':')) {
            throw new Error(util_1.PRESET_PROHIBITED_SUBPRESET);
        }
        if (!nonScopedPresetWithSubdirRegex.test(str)) {
            throw new Error(util_1.PRESET_INVALID);
        }
        ({ repo, presetPath, presetName, tag } =
            nonScopedPresetWithSubdirRegex.exec(str).groups);
    }
    else {
        ({ repo, presetName, tag } = gitPresetRegex.exec(str).groups);
        if (presetSource === 'npm' && !repo.startsWith('renovate-config-')) {
            repo = `renovate-config-${repo}`;
        }
        if (!is_1.default.nonEmptyString(presetName)) {
            presetName = 'default';
        }
    }
    return {
        presetSource,
        presetPath,
        repo,
        presetName,
        tag,
        params,
        rawParams,
    };
}
//# sourceMappingURL=parse.js.map