"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HostRulesMigration = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const url_1 = require("../../../util/url");
const abstract_migration_1 = require("../base/abstract-migration");
const datasource_migration_1 = require("./datasource-migration");
class HostRulesMigration extends abstract_migration_1.AbstractMigration {
    propertyName = 'hostRules';
    run(value) {
        const newHostRules = [];
        for (const hostRule of value) {
            validateHostRule(hostRule);
            const newRule = {};
            for (const [key, value] of Object.entries(hostRule)) {
                if (key === 'platform') {
                    if (is_1.default.string(value)) {
                        newRule.hostType ??= value;
                    }
                    continue;
                }
                if (key === 'matchHost') {
                    if (is_1.default.string(value)) {
                        newRule.matchHost ??= (0, url_1.massageHostUrl)(value);
                    }
                    continue;
                }
                if (key === 'hostType') {
                    if (is_1.default.string(value)) {
                        newRule.hostType ??= (0, datasource_migration_1.migrateDatasource)(value);
                    }
                    continue;
                }
                if (key === 'endpoint' ||
                    key === 'host' ||
                    key === 'baseUrl' ||
                    key === 'hostName' ||
                    key === 'domainName') {
                    if (is_1.default.string(value)) {
                        newRule.matchHost ??= (0, url_1.massageHostUrl)(value);
                    }
                    continue;
                }
                newRule[key] = value;
            }
            newHostRules.push(newRule);
        }
        this.rewrite(newHostRules);
    }
}
exports.HostRulesMigration = HostRulesMigration;
function validateHostRule(rule) {
    const { matchHost, hostName, domainName, baseUrl, endpoint, host } = rule;
    const hosts = removeUndefinedFields({
        matchHost,
        hostName,
        domainName,
        baseUrl,
        endpoint,
        host,
    });
    if (Object.keys(hosts).length > 1) {
        const distinctHostValues = new Set(Object.values(hosts));
        // check if the host values are duplicated
        if (distinctHostValues.size > 1) {
            const error = new Error(error_messages_1.CONFIG_VALIDATION);
            error.validationSource = 'config';
            error.validationMessage =
                '`hostRules` cannot contain more than one host-matching field - use `matchHost` only.';
            error.validationError =
                'The renovate configuration file contains some invalid settings';
            throw error;
        }
        else {
            logger_1.logger.warn({ hosts }, 'Duplicate host values found, please only use `matchHost` to specify the host');
        }
    }
}
function removeUndefinedFields(obj) {
    const result = {};
    for (const key of Object.keys(obj)) {
        if (is_1.default.string(obj[key])) {
            result[key] = obj[key];
        }
    }
    return result;
}
//# sourceMappingURL=host-rules-migration.js.map