"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.tryDecryptOpenPgp = tryDecryptOpenPgp;
const expose_cjs_1 = require("../../expose.cjs");
const logger_1 = require("../../logger");
const regex_1 = require("../../util/regex");
let pgp = undefined;
async function tryDecryptOpenPgp(privateKey, encryptedStr) {
    if (encryptedStr.length < 500) {
        // optimization during transition of public key -> pgp
        return null;
    }
    if (pgp === undefined) {
        try {
            pgp = (0, expose_cjs_1.openpgp)();
        }
        catch (err) {
            logger_1.logger.warn({ err }, 'Could load openpgp');
            pgp = null;
        }
    }
    if (pgp === null) {
        logger_1.logger.once.warn('Cannot load openpgp, skipping decryption');
        return null;
    }
    try {
        const pk = await pgp.readPrivateKey({
            // prettier-ignore
            armoredKey: privateKey.replace((0, regex_1.regEx)(/\n[ \t]+/g), '\n'), // little massage to help a common problem
        });
        const startBlock = '-----BEGIN PGP MESSAGE-----\n\n';
        const endBlock = '\n-----END PGP MESSAGE-----';
        let armoredMessage = encryptedStr.trim();
        if (!armoredMessage.startsWith(startBlock)) {
            armoredMessage = `${startBlock}${armoredMessage}`;
        }
        if (!armoredMessage.endsWith(endBlock)) {
            armoredMessage = `${armoredMessage}${endBlock}`;
        }
        const message = await pgp.readMessage({
            armoredMessage,
        });
        const { data } = await pgp.decrypt({
            message,
            decryptionKeys: pk,
        });
        logger_1.logger.debug('Decrypted config using openpgp');
        return data;
    }
    catch (err) {
        logger_1.logger.debug({ err }, 'Could not decrypt using openpgp');
        return null;
    }
}
//# sourceMappingURL=openpgp.js.map