"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setPrivateKeys = setPrivateKeys;
exports.tryDecrypt = tryDecrypt;
exports.validateDecryptedValue = validateDecryptedValue;
exports.decryptConfig = decryptConfig;
exports.getAzureCollection = getAzureCollection;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const error_messages_1 = require("../constants/error-messages");
const logger_1 = require("../logger");
const env_1 = require("../util/env");
const regex_1 = require("../util/regex");
const sanitize_1 = require("../util/sanitize");
const url_1 = require("../util/url");
const kbpgp_1 = require("./decrypt/kbpgp");
const legacy_1 = require("./decrypt/legacy");
const openpgp_1 = require("./decrypt/openpgp");
const global_1 = require("./global");
const schema_1 = require("./schema");
let privateKey;
let privateKeyOld;
function setPrivateKeys(pKey, pKeyOld) {
    privateKey = pKey;
    privateKeyOld = pKeyOld;
}
async function tryDecrypt(key, encryptedStr, repository, keyName) {
    let decryptedStr = null;
    if (key?.startsWith('-----BEGIN PGP PRIVATE KEY BLOCK-----')) {
        const decryptedObjStr = (0, env_1.getEnv)().RENOVATE_X_USE_OPENPGP === 'true'
            ? await (0, openpgp_1.tryDecryptOpenPgp)(key, encryptedStr)
            : await (0, kbpgp_1.tryDecryptKbPgp)(key, encryptedStr);
        if (decryptedObjStr) {
            decryptedStr = validateDecryptedValue(decryptedObjStr, repository);
        }
    }
    else {
        decryptedStr = (0, legacy_1.tryDecryptPublicKeyDefault)(key, encryptedStr);
        if (is_1.default.string(decryptedStr)) {
            logger_1.logger.warn({ keyName }, 'Encrypted value is using deprecated default padding, please change to using PGP encryption.');
        }
        else {
            decryptedStr = (0, legacy_1.tryDecryptPublicKeyPKCS1)(key, encryptedStr);
            /* v8 ignore start -- not testable */
            if (is_1.default.string(decryptedStr)) {
                logger_1.logger.warn({ keyName }, 'Encrypted value is using deprecated PKCS1 padding, please change to using PGP encryption.');
            }
            /* v8 ignore stop */
        }
    }
    return decryptedStr;
}
function validateDecryptedValue(decryptedObjStr, repository) {
    try {
        const decryptedObj = schema_1.DecryptedObject.safeParse(decryptedObjStr);
        if (!decryptedObj.success) {
            const error = new Error('config-validation');
            error.validationError = `Could not parse decrypted config.`;
            throw error;
        }
        const { o: org, r: repo, v: value } = decryptedObj.data;
        if (!is_1.default.nonEmptyString(value)) {
            const error = new Error('config-validation');
            error.validationError = `Encrypted value in config is missing a value.`;
            throw error;
        }
        if (!is_1.default.nonEmptyString(org)) {
            const error = new Error('config-validation');
            error.validationError = `Encrypted value in config is missing a scope.`;
            throw error;
        }
        const repositories = [repository.toUpperCase()];
        const azureCollection = getAzureCollection();
        if (is_1.default.nonEmptyString(azureCollection)) {
            // used for full 'org/project/repo' matching
            repositories.push(`${azureCollection}/${repository}`.toUpperCase());
            // used for org prefix matching without repo
            repositories.push(`${azureCollection}/*/`.toUpperCase());
        }
        const orgPrefixes = org
            .split(',')
            .map((o) => o.trim())
            .map((o) => o.toUpperCase())
            .map((o) => (0, url_1.ensureTrailingSlash)(o));
        if (is_1.default.nonEmptyString(repo)) {
            const scopedRepos = orgPrefixes.map((orgPrefix) => `${orgPrefix}${repo}`.toUpperCase());
            for (const rp of repositories) {
                if (scopedRepos.some((r) => r === rp)) {
                    return value;
                }
            }
            logger_1.logger.debug({ scopedRepos }, 'Secret is scoped to a different repository');
            const error = new Error('config-validation');
            const scopeString = scopedRepos.join(',');
            error.validationError = `Encrypted secret is scoped to a different repository: "${scopeString}".`;
            throw error;
        }
        // no scoped repos, only org
        const azcol = azureCollection === undefined
            ? undefined
            : (0, url_1.ensureTrailingSlash)(azureCollection).toUpperCase();
        for (const rp of repositories) {
            if (orgPrefixes.some((orgPrefix) => rp.startsWith(orgPrefix) && orgPrefix !== azcol)) {
                return value;
            }
        }
        logger_1.logger.debug({ orgPrefixes }, 'Secret is scoped to a different org');
        const error = new Error('config-validation');
        const scopeString = orgPrefixes.join(',');
        error.validationError = `Encrypted secret is scoped to a different org: "${scopeString}".`;
        throw error;
    }
    catch (err) {
        logger_1.logger.warn({ err }, 'Could not parse decrypted string');
    }
    return null;
}
async function decryptConfig(config, repository, existingPath = '$') {
    logger_1.logger.trace({ config }, 'decryptConfig()');
    const decryptedConfig = { ...config };
    for (const [key, val] of Object.entries(config)) {
        if (key === 'encrypted' && is_1.default.object(val)) {
            const path = `${existingPath}.${key}`;
            logger_1.logger.debug({ config: val }, `Found encrypted config in ${path}`);
            const encryptedWarning = global_1.GlobalConfig.get('encryptedWarning');
            if (is_1.default.string(encryptedWarning)) {
                logger_1.logger.once.warn(encryptedWarning);
            }
            if (privateKey) {
                for (const [eKey, eVal] of Object.entries(val)) {
                    logger_1.logger.debug(`Trying to decrypt ${eKey} in ${path}`);
                    let decryptedStr = await tryDecrypt(privateKey, eVal, repository, eKey);
                    if (privateKeyOld && !is_1.default.nonEmptyString(decryptedStr)) {
                        logger_1.logger.debug(`Trying to decrypt with old private key`);
                        decryptedStr = await tryDecrypt(privateKeyOld, eVal, repository, eKey);
                    }
                    if (!is_1.default.nonEmptyString(decryptedStr)) {
                        const error = new Error('config-validation');
                        error.validationError = `Failed to decrypt field ${eKey}. Please re-encrypt and try again.`;
                        throw error;
                    }
                    logger_1.logger.debug(`Decrypted ${eKey} in ${path}`);
                    if (eKey === 'npmToken') {
                        const token = decryptedStr.replace((0, regex_1.regEx)(/\n$/), '');
                        decryptedConfig[eKey] = token;
                        (0, sanitize_1.addSecretForSanitizing)(token);
                    }
                    else {
                        decryptedConfig[eKey] = decryptedStr;
                        (0, sanitize_1.addSecretForSanitizing)(decryptedStr);
                    }
                }
            }
            else {
                const env = (0, env_1.getEnv)();
                if (env.RENOVATE_X_ENCRYPTED_STRICT === 'true') {
                    const error = new Error(error_messages_1.CONFIG_VALIDATION);
                    error.validationSource = 'config';
                    error.validationError = 'Encrypted config unsupported';
                    error.validationMessage = `This config contains an encrypted object at location \`$.${key}\` but no privateKey is configured. To support encrypted config, the Renovate administrator must configure a \`privateKey\` in Global Configuration.`;
                    if (env.MEND_HOSTED === 'true') {
                        error.validationMessage = `Mend-hosted Renovate Apps no longer support the use of encrypted secrets in Renovate file config (e.g. renovate.json).
Please migrate all secrets to the Developer Portal using the web UI available at https://developer.mend.io/

Refer to migration documents here: https://docs.renovatebot.com/mend-hosted/migrating-secrets/`;
                    }
                    throw error;
                }
                else {
                    logger_1.logger.error('Found encrypted data but no privateKey');
                }
            }
            delete decryptedConfig.encrypted;
        }
        else if (is_1.default.array(val)) {
            decryptedConfig[key] = [];
            for (const [index, item] of val.entries()) {
                if (is_1.default.object(item) && !is_1.default.array(item)) {
                    const path = `${existingPath}.${key}[${index}]`;
                    decryptedConfig[key].push(await decryptConfig(item, repository, path));
                }
                else {
                    decryptedConfig[key].push(item);
                }
            }
        }
        else if (is_1.default.object(val) && key !== 'content') {
            const path = `${existingPath}.${key}`;
            decryptedConfig[key] = await decryptConfig(val, repository, path);
        }
    }
    delete decryptedConfig.encrypted;
    logger_1.logger.trace({ config: decryptedConfig }, 'decryptedConfig');
    return decryptedConfig;
}
function getAzureCollection() {
    const platform = global_1.GlobalConfig.get('platform');
    if (platform !== 'azure') {
        return undefined;
    }
    const endpoint = global_1.GlobalConfig.get('endpoint');
    const endpointUrl = (0, url_1.parseUrl)(endpoint);
    if (endpointUrl === null) {
        // should not happen
        logger_1.logger.warn({ endpoint }, 'Unable to parse endpoint for token decryption');
        return undefined;
    }
    const azureCollection = (0, url_1.trimSlashes)(endpointUrl.pathname);
    if (!is_1.default.nonEmptyString(azureCollection)) {
        logger_1.logger.debug({ endpoint }, 'Unable to find azure collection name from URL');
        return undefined;
    }
    if (azureCollection.startsWith('tfs/')) {
        // Azure DevOps Server
        return azureCollection.substring(4);
    }
    return azureCollection;
}
//# sourceMappingURL=decrypt.js.map