#!/usr/bin/env node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("source-map-support/register");
require("./punycode.cjs");
const dequal_1 = require("dequal");
const fs_extra_1 = require("fs-extra");
const app_strings_1 = require("./config/app-strings");
const massage_1 = require("./config/massage");
const migration_1 = require("./config/migration");
const validation_1 = require("./config/validation");
const logger_1 = require("./logger");
const env_1 = require("./util/env");
const file_1 = require("./workers/global/config/parse/file");
let returnVal = 0;
async function validate(configType, desc, config, strict, isPreset = false) {
    const { isMigrated, migratedConfig } = (0, migration_1.migrateConfig)(config);
    if (isMigrated) {
        logger_1.logger.warn({
            oldConfig: config,
            newConfig: migratedConfig,
        }, 'Config migration necessary');
        if (strict) {
            returnVal = 1;
        }
    }
    const massagedConfig = (0, massage_1.massageConfig)(migratedConfig);
    const res = await (0, validation_1.validateConfig)(configType, massagedConfig, isPreset);
    if (res.errors.length) {
        logger_1.logger.error({ file: desc, errors: res.errors }, 'Found errors in configuration');
        returnVal = 1;
    }
    if (res.warnings.length) {
        logger_1.logger.warn({ file: desc, warnings: res.warnings }, 'Found errors in configuration');
        returnVal = 1;
    }
}
(async () => {
    const strictArgIndex = process.argv.indexOf('--strict');
    const strict = strictArgIndex >= 0;
    if (strict) {
        process.argv.splice(strictArgIndex, 1);
    }
    if (process.argv.length > 2) {
        for (const file of process.argv.slice(2)) {
            try {
                if (!(await (0, fs_extra_1.pathExists)(file))) {
                    returnVal = 1;
                    logger_1.logger.error({ file }, 'File does not exist');
                    break;
                }
                const parsedContent = await (0, file_1.getParsedContent)(file);
                try {
                    logger_1.logger.info(`Validating ${file}`);
                    await validate('global', file, parsedContent, strict);
                }
                catch (err) {
                    logger_1.logger.warn({ file, err }, 'File is not valid Renovate config');
                    returnVal = 1;
                }
            }
            catch (err) {
                logger_1.logger.warn({ file, err }, 'File could not be parsed');
                returnVal = 1;
            }
        }
    }
    else {
        for (const file of app_strings_1.configFileNames.filter((name) => name !== 'package.json')) {
            try {
                if (!(await (0, fs_extra_1.pathExists)(file))) {
                    continue;
                }
                const parsedContent = await (0, file_1.getParsedContent)(file);
                try {
                    logger_1.logger.info(`Validating ${file}`);
                    await validate('repo', file, parsedContent, strict);
                }
                catch (err) {
                    logger_1.logger.warn({ file, err }, 'File is not valid Renovate config');
                    returnVal = 1;
                }
            }
            catch (err) {
                logger_1.logger.warn({ file, err }, 'File could not be parsed');
                returnVal = 1;
            }
        }
        try {
            const pkgJson = JSON.parse(await (0, fs_extra_1.readFile)('package.json', 'utf8'));
            if (pkgJson.renovate) {
                logger_1.logger.info(`Validating package.json > renovate`);
                await validate('repo', 'package.json > renovate', pkgJson.renovate, strict);
            }
            if (pkgJson['renovate-config']) {
                logger_1.logger.info(`Validating package.json > renovate-config`);
                for (const presetConfig of Object.values(pkgJson['renovate-config'])) {
                    await validate('repo', 'package.json > renovate-config', presetConfig, strict, true);
                }
            }
        }
        catch {
            // ignore
        }
        try {
            const env = (0, env_1.getEnv)();
            const fileConfig = await (0, file_1.getConfig)(env);
            if (!(0, dequal_1.dequal)(fileConfig, {})) {
                const file = env.RENOVATE_CONFIG_FILE ?? 'config.js';
                logger_1.logger.info(`Validating ${file}`);
                try {
                    await validate('global', file, fileConfig, strict);
                }
                catch (err) {
                    logger_1.logger.error({ file, err }, 'File is not valid Renovate config');
                    returnVal = 1;
                }
            }
        }
        catch {
            // ignore
        }
    }
    if (returnVal !== 0) {
        process.exit(returnVal);
    }
    logger_1.logger.info('Config validated successfully');
})().catch((e) => {
    // eslint-disable-next-line no-console
    console.error(e);
    process.exit(99);
});
//# sourceMappingURL=config-validator.js.map