import { compile as mooCompile, states as mooStates, } from 'moo';
import { configBrackets } from './bracket';
import { configComments } from './comment';
import { configNumbers } from './number';
import { configOperators } from './operator';
import { createOrderedStateMap, fallbackRule } from './rules';
import { configStrings } from './string';
import { configSymbols } from './symbol';
import { coerceToken } from './token';
export * from './token';
export * from './types';
export function configureLexerRules(lexerConfig) {
    const whitespace = lexerConfig.joinLines
        ? {
            t: 'regex',
            type: 'whitespace',
            match: new RegExp(`(?:${lexerConfig.joinLines}\\r?\\n|[ \\t\\r])+`),
            lineBreaks: true,
            chunk: null,
        }
        : {
            t: 'regex',
            type: 'whitespace',
            match: /[ \t\r]+/,
            chunk: null,
        };
    let result = {
        $: {
            whitespace,
            newline: {
                t: 'regex',
                type: 'newline',
                match: /\r?\n/,
                chunk: null,
                lineBreaks: true,
            },
            _: { ...fallbackRule, type: '_' },
        },
    };
    const { comments, symbols, operators, brackets, strings, numbers } = lexerConfig;
    result = configComments(result, comments);
    result = configSymbols(result, { match: symbols });
    result = configOperators(result, operators);
    result = configBrackets(result, brackets);
    result = configStrings(result, strings);
    result = configNumbers(result, { match: numbers });
    const orderedResult = createOrderedStateMap(result);
    return orderedResult;
}
function isVarToken(key) {
    return typeof key === 'string' && /^str\$\d+\$tpl\$\d+\$token$/.test(key);
}
function getVarEndToken({ type, value, offset, line, col, lineBreaks, }) {
    return {
        type: type === null || type === void 0 ? void 0 : type.replace(/\$[^$]+$/, '$end'),
        value: '',
        text: '',
        offset: offset + value.length,
        line: line + lineBreaks,
        col: col + value.length,
        lineBreaks: 0,
    };
}
function getSubLexers(states) {
    const result = {};
    for (const [key, rules] of Object.entries(states)) {
        if (isVarToken(key)) {
            result[key] = mooCompile(rules);
        }
    }
    return result;
}
export function createLexer(options) {
    const rules = configureLexerRules(options);
    const subLexers = getSubLexers(rules);
    const mainLexer = mooStates(rules);
    let subLexer;
    let subLexerToken;
    const result = {
        reset(input) {
            Object.values(subLexers).forEach((subLexer) => subLexer.reset());
            subLexer = undefined;
            subLexerToken = undefined;
            if (typeof input === 'undefined' || input === null) {
                mainLexer.reset();
            }
            else {
                mainLexer.reset(input);
            }
            return result;
        },
        [Symbol.iterator]() {
            const next = () => {
                const mooLexer = subLexer !== null && subLexer !== void 0 ? subLexer : mainLexer;
                const mooIter = mooLexer[Symbol.iterator]();
                const nextElem = mooIter.next();
                if (nextElem.done) {
                    if (subLexer && subLexerToken) {
                        const subLexerType = subLexerToken.type;
                        mooLexer.reset();
                        if (isVarToken(subLexerType)) {
                            const x = getVarEndToken(subLexerToken);
                            const value = coerceToken(x);
                            subLexer = undefined;
                            subLexerToken = undefined;
                            return { done: false, value };
                        }
                        else {
                            subLexer = undefined;
                            subLexerToken = undefined;
                            return next();
                        }
                    }
                    else {
                        return { done: true, value: null };
                    }
                }
                const mooToken = nextElem.value;
                if (!subLexer) {
                    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                    const mooTokenType = mooToken.type;
                    subLexer = subLexers[mooTokenType];
                    if (subLexer) {
                        subLexerToken = mooToken;
                        subLexer.reset(mooToken.value);
                        return next();
                    }
                }
                const value = coerceToken(mooToken, subLexerToken);
                return { done: false, value };
            };
            return { next };
        },
    };
    return result;
}
//# sourceMappingURL=index.js.map