"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TreeMatcher = void 0;
const parser_1 = require("../../parser");
const handler_1 = require("../handler");
const abstract_matcher_1 = require("./abstract-matcher");
class TreeMatcher extends abstract_matcher_1.AbstractMatcher {
    constructor(config) {
        var _a, _b, _c;
        super();
        this.walkDepth = 0;
        this.matchCount = 0;
        this.type = (_a = config.type) !== null && _a !== void 0 ? _a : null;
        this.startsWithValue = (_b = config.startsWith) !== null && _b !== void 0 ? _b : null;
        this.endsWithValue = (_c = config.endsWith) !== null && _c !== void 0 ? _c : null;
        this.matcher = config.matcher;
        this.maxDepth =
            typeof config.maxDepth === 'number' && config.maxDepth > 0
                ? config.maxDepth
                : 1024;
        this.maxMatches =
            typeof config.maxMatches === 'number' && config.maxMatches > 0
                ? config.maxMatches
                : 1024 * 1024;
        this.preHandler = (0, handler_1.safeHandler)(config.preHandler);
        this.postHandler = (0, handler_1.safeHandler)(config.postHandler);
    }
    walkToNextSignificantNode(cursor) {
        const downCursor = cursor.down;
        if (downCursor && this.walkDepth < this.maxDepth) {
            this.walkDepth += 1;
            return downCursor;
        }
        let rightCursor = cursor.right;
        if (rightCursor) {
            rightCursor = this.matcher
                ? this.matcher.seekNext(rightCursor)
                : this.seekNext(rightCursor);
            if (rightCursor) {
                return rightCursor;
            }
        }
        const canMoveUp = this.walkDepth > 1;
        let upperCursor = canMoveUp ? cursor.up : undefined;
        while (upperCursor && canMoveUp) {
            rightCursor = upperCursor.right;
            if (rightCursor) {
                rightCursor = this.matcher
                    ? this.matcher.seekNext(rightCursor)
                    : this.seekNext(rightCursor);
                if (rightCursor) {
                    upperCursor = rightCursor;
                    break;
                }
            }
            upperCursor = upperCursor.up;
        }
        if (upperCursor) {
            this.walkDepth -= 1;
            return upperCursor;
        }
        return undefined;
    }
    walkToNextMatch(context, cursor) {
        if (!cursor || !this.matcher) {
            return undefined;
        }
        let nextCursor = this.walkToNextSignificantNode(cursor);
        while (nextCursor) {
            const match = this.matcher.match({ cursor: nextCursor, context });
            if (match) {
                this.matchCount += 1;
                return match;
            }
            nextCursor = this.walkToNextSignificantNode(nextCursor);
        }
        return undefined;
    }
    match(checkpoint) {
        this.walkDepth = 0;
        this.matchCount = 0;
        const rootCursor = this.seekNext(checkpoint.cursor);
        const rootNode = rootCursor.node;
        if ((0, parser_1.isTree)(rootNode)) {
            if (this.type && this.type !== rootNode.type) {
                return null;
            }
            if (this.type === 'wrapped-tree') {
                const currentTree = rootNode;
                if (this.startsWithValue &&
                    currentTree.startsWith.value !== this.startsWithValue) {
                    return null;
                }
                if (this.endsWithValue &&
                    currentTree.endsWith.value !== this.endsWithValue) {
                    return null;
                }
            }
            let context = checkpoint.context;
            context = this.preHandler(context, rootNode);
            if (this.matcher) {
                let nextMatch = this.walkToNextMatch(context, rootCursor);
                while (nextMatch) {
                    context = nextMatch.context;
                    if (this.matchCount === this.maxMatches) {
                        break;
                    }
                    nextMatch = this.walkToNextMatch(context, nextMatch.cursor);
                }
                if (this.matchCount === 0) {
                    return null;
                }
            }
            context = this.postHandler(context, rootNode);
            const cursor = rootNode.type === 'root-tree' ? rootCursor : this.moveRight(rootCursor);
            return { context, cursor };
        }
        return null;
    }
}
exports.TreeMatcher = TreeMatcher;
//# sourceMappingURL=tree-matcher.js.map