import { join } from 'path';
import { fileURLToPath } from 'url';
import { readFile } from 'fs/promises';
import Handlebars from 'handlebars';
import { filterRevertedCommitsSync } from 'conventional-commits-filter';
import { getTemplateContext } from './context.js';
const dirname = fileURLToPath(new URL('.', import.meta.url));
/**
 * Load templates from files.
 * @param options
 * @returns Templates strings object.
 */
export async function loadTemplates(options = {}) {
    const [mainTemplate, headerPartial, commitPartial, footerPartial] = await Promise.all([
        options.mainTemplate || readFile(join(dirname, '..', 'templates', 'template.hbs'), 'utf-8'),
        options.headerPartial || readFile(join(dirname, '..', 'templates', 'header.hbs'), 'utf-8'),
        options.commitPartial || readFile(join(dirname, '..', 'templates', 'commit.hbs'), 'utf-8'),
        options.footerPartial || readFile(join(dirname, '..', 'templates', 'footer.hbs'), 'utf-8')
    ]);
    return {
        mainTemplate,
        headerPartial,
        commitPartial,
        footerPartial
    };
}
/**
 * Compile Handlebars templates.
 * @param templates
 * @returns Handlebars template instance.
 */
export function compileTemplates(templates) {
    const { mainTemplate, headerPartial, commitPartial, footerPartial, partials } = templates;
    Handlebars.registerPartial('header', headerPartial);
    Handlebars.registerPartial('commit', commitPartial);
    Handlebars.registerPartial('footer', footerPartial);
    if (partials) {
        Object.entries(partials).forEach(([name, partial]) => {
            if (typeof partial === 'string') {
                Handlebars.registerPartial(name, partial);
            }
        });
    }
    return Handlebars.compile(mainTemplate, {
        noEscape: true
    });
}
/**
 * Create template renderer.
 * @param context
 * @param options
 * @returns Template render function.
 */
export function createTemplateRenderer(context, options) {
    const { ignoreReverted } = options;
    const template = compileTemplates(options);
    return async (commits, keyCommit) => {
        const notes = [];
        const commitsForTemplate = (ignoreReverted
            ? Array.from(filterRevertedCommitsSync(commits))
            : commits).map(commit => ({
            ...commit,
            notes: commit.notes.map((note) => {
                const commitNote = {
                    ...note,
                    commit
                };
                notes.push(commitNote);
                return commitNote;
            })
        }));
        const templateContext = await getTemplateContext(keyCommit, commits, commitsForTemplate, notes, context, options);
        return template(templateContext);
    };
}
//# sourceMappingURL=data:application/json;base64,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