/// <reference types="node" />
import basem = require('./ClientApiBases');
import VsoBaseInterfaces = require('./interfaces/common/VsoBaseInterfaces');
import AlertInterfaces = require("./interfaces/AlertInterfaces");
import VSSInterfaces = require("./interfaces/common/VSSInterfaces");
export interface IAlertApi extends basem.ClientApiBase {
    getAlert(project: string, alertId: number, repository: string, ref?: string, expand?: AlertInterfaces.ExpandOption): Promise<AlertInterfaces.Alert>;
    getAlerts(project: string, repository: string, top?: number, orderBy?: string, criteria?: AlertInterfaces.SearchCriteria, expand?: AlertInterfaces.AlertListExpandOption, continuationToken?: string): Promise<VSSInterfaces.PagedList<AlertInterfaces.Alert>>;
    getAlertSarif(project: string, alertId: number, repository: string, ref?: string, expand?: AlertInterfaces.ExpandOption): Promise<string>;
    updateAlert(stateUpdate: AlertInterfaces.AlertStateUpdate, project: string, alertId: number, repository: string): Promise<AlertInterfaces.Alert>;
    getBranches(project: string, repository: string, alertType: AlertInterfaces.AlertType, continuationToken?: string, branchNameContains?: string, top?: number, includePullRequestBranches?: boolean): Promise<AlertInterfaces.Branch[]>;
    getUxFilters(project: string, repository: string, alertType: AlertInterfaces.AlertType): Promise<AlertInterfaces.UxFilters>;
    getAlertInstances(project: string, alertId: number, repository: string, ref?: string): Promise<AlertInterfaces.AlertAnalysisInstance[]>;
    createLegalReview(project: string, repository: string, alertId: number, ref?: string): Promise<AlertInterfaces.LegalReview>;
    updateAlertsMetadata(alertsMetadata: AlertInterfaces.AlertMetadata[], project: string, repository: string): Promise<AlertInterfaces.AlertMetadataChange[]>;
    uploadSarif(customHeaders: any, contentStream: NodeJS.ReadableStream, project: string, repository: string, notificationFlag?: String): Promise<number>;
    getSarif(sarifId: number): Promise<AlertInterfaces.SarifUploadStatus>;
    getValidityData(project: string, repository: string, alertId: number): Promise<AlertInterfaces.ValidationRequestInfo>;
    initiateValidation(project: string, repository: string, alertId: number): Promise<AlertInterfaces.AlertValidationRequestStatus>;
}
export declare class AlertApi extends basem.ClientApiBase implements IAlertApi {
    constructor(baseUrl: string, handlers: VsoBaseInterfaces.IRequestHandler[], options?: VsoBaseInterfaces.IRequestOptions, userAgent?: string);
    /**
     * Get an alert.
     *
     * @param {string} project - Project ID or project name
     * @param {number} alertId - ID of alert to retrieve
     * @param {string} repository - Name or id  of a repository that alert is part of
     * @param {string} ref
     * @param {AlertInterfaces.ExpandOption} expand - Expand alert attributes. Possible options are {ValidationFingerprint, None}
     */
    getAlert(project: string, alertId: number, repository: string, ref?: string, expand?: AlertInterfaces.ExpandOption): Promise<AlertInterfaces.Alert>;
    /**
     * Get alerts for a repository
     *
     * @param {string} project - Project ID or project name
     * @param {string} repository - The name or ID of the repository
     * @param {number} top - The maximum number of alerts to return
     * @param {string} orderBy - Must be "id" "firstSeen" "lastSeen" "fixedOn" or "severity"  Defaults to "id"
     * @param {AlertInterfaces.SearchCriteria} criteria - Options to limit the alerts returned
     * @param {AlertInterfaces.AlertListExpandOption} expand
     * @param {string} continuationToken - If there are more alerts than can be returned, a continuation token is placed in the "x-ms-continuationtoken" header.  Use that token here to get the next page of alerts
     */
    getAlerts(project: string, repository: string, top?: number, orderBy?: string, criteria?: AlertInterfaces.SearchCriteria, expand?: AlertInterfaces.AlertListExpandOption, continuationToken?: string): Promise<VSSInterfaces.PagedList<AlertInterfaces.Alert>>;
    /**
     * Get an alert.
     *
     * @param {string} project - Project ID or project name
     * @param {number} alertId - ID of alert to retrieve
     * @param {string} repository - Name or id  of a repository that alert is part of
     * @param {string} ref
     * @param {AlertInterfaces.ExpandOption} expand - Expand alert attributes. Possible options are {ValidationFingerprint, None}
     */
    getAlertSarif(project: string, alertId: number, repository: string, ref?: string, expand?: AlertInterfaces.ExpandOption): Promise<string>;
    /**
     * Update the status of an alert
     *
     * @param {AlertInterfaces.AlertStateUpdate} stateUpdate - The new status of the alert
     * @param {string} project - Project ID or project name
     * @param {number} alertId - The ID of the alert
     * @param {string} repository - The name or ID of the repository
     */
    updateAlert(stateUpdate: AlertInterfaces.AlertStateUpdate, project: string, alertId: number, repository: string): Promise<AlertInterfaces.Alert>;
    /**
     * Returns the branches for which analysis results were submitted.
     *
     * @param {string} project - Project ID or project name
     * @param {string} repository
     * @param {AlertInterfaces.AlertType} alertType - The type of alert: Dependency Scanning (1), Secret (2), Code QL (3), etc.
     * @param {string} continuationToken - A string variable that represents the branch name and is used to fetch branches that follow it in alphabetical order.
     * @param {string} branchNameContains - A string variable used to fetch branches that contain this string anywhere in the branch name, case insensitive.
     * @param {number} top - An int variable used to return the top k branches that satisfy the search criteria.
     * @param {boolean} includePullRequestBranches - A bool variable indicating whether or not to include pull request branches.
     */
    getBranches(project: string, repository: string, alertType: AlertInterfaces.AlertType, continuationToken?: string, branchNameContains?: string, top?: number, includePullRequestBranches?: boolean): Promise<AlertInterfaces.Branch[]>;
    /**
     * @param {string} project - Project ID or project name
     * @param {string} repository
     * @param {AlertInterfaces.AlertType} alertType
     */
    getUxFilters(project: string, repository: string, alertType: AlertInterfaces.AlertType): Promise<AlertInterfaces.UxFilters>;
    /**
     * Get instances of an alert on a branch specified with @ref. If @ref is not provided, return instances of an alert on default branch(if the alert exist in default branch) or latest affected branch.
     *
     * @param {string} project - Project ID or project name
     * @param {number} alertId - ID of alert to retrieve
     * @param {string} repository - Name or id of a repository that alert is part of
     * @param {string} ref
     */
    getAlertInstances(project: string, alertId: number, repository: string, ref?: string): Promise<AlertInterfaces.AlertAnalysisInstance[]>;
    /**
     * Create legal review. This creates the legal review associated with the alert. It include the review work item url.
     *
     * @param {string} project - Project ID or project name
     * @param {string} repository - Name or id  of a repository for the legal alert
     * @param {number} alertId - Advance Security alert id of the legal alert to get the legal review
     * @param {string} ref
     */
    createLegalReview(project: string, repository: string, alertId: number, ref?: string): Promise<AlertInterfaces.LegalReview>;
    /**
     * Update alert metadata associations.
     *
     * @param {AlertInterfaces.AlertMetadata[]} alertsMetadata - A list of metadata to associate with alerts.
     * @param {string} project - Project ID or project name
     * @param {string} repository - The name or ID of the repository.
     */
    updateAlertsMetadata(alertsMetadata: AlertInterfaces.AlertMetadata[], project: string, repository: string): Promise<AlertInterfaces.AlertMetadataChange[]>;
    /**
     * Upload a Sarif containing security alerts
     *
     * @param {NodeJS.ReadableStream} contentStream - Content to upload
     * @param {string} project - Project ID or project name
     * @param {string} repository - The name or ID of a repository
     * @param {String} notificationFlag - Header to signal that this is a progress notification
     */
    uploadSarif(customHeaders: any, contentStream: NodeJS.ReadableStream, project: string, repository: string, notificationFlag?: String): Promise<number>;
    /**
     * Get the status of the Sarif processing job
     *
     * @param {number} sarifId - Sarif ID returned when the Sarif was uploaded
     */
    getSarif(sarifId: number): Promise<AlertInterfaces.SarifUploadStatus>;
    /**
     * Get the validity details for an alert.
     *
     * @param {string} project - Project ID or project name
     * @param {string} repository - The name or ID of a repository
     * @param {number} alertId - The ID of the alert
     */
    getValidityData(project: string, repository: string, alertId: number): Promise<AlertInterfaces.ValidationRequestInfo>;
    /**
     * Initiate the validation process for a given alert
     *
     * @param {string} project - Project ID or project name
     * @param {string} repository - The name or ID of a repository
     * @param {number} alertId - The ID of the alert
     */
    initiateValidation(project: string, repository: string, alertId: number): Promise<AlertInterfaces.AlertValidationRequestStatus>;
}
