"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ZipFS = exports.DEFAULT_COMPRESSION_LEVEL = exports.DEFLATE = exports.STORE = exports.ZIP_UNIX = void 0;
exports.makeEmptyArchive = makeEmptyArchive;
const tslib_1 = require("tslib");
const fslib_1 = require("@yarnpkg/fslib");
const fslib_2 = require("@yarnpkg/fslib");
const fslib_3 = require("@yarnpkg/fslib");
const fslib_4 = require("@yarnpkg/fslib");
const fslib_5 = require("@yarnpkg/fslib");
const fslib_6 = require("@yarnpkg/fslib");
const fs_1 = require("fs");
const stream_1 = require("stream");
const util_1 = require("util");
const zlib_1 = tslib_1.__importDefault(require("zlib"));
const libzipImpl_1 = require("./libzipImpl");
exports.ZIP_UNIX = 3;
exports.STORE = 0;
exports.DEFLATE = 8;
exports.DEFAULT_COMPRESSION_LEVEL = `mixed`;
function toUnixTimestamp(time) {
    if (typeof time === `string` && String(+time) === time)
        return +time;
    if (typeof time === `number` && Number.isFinite(time)) {
        if (time < 0) {
            return Date.now() / 1000;
        }
        else {
            return time;
        }
    }
    // convert to 123.456 UNIX timestamp
    if (util_1.types.isDate(time))
        return time.getTime() / 1000;
    throw new Error(`Invalid time`);
}
function makeEmptyArchive() {
    return Buffer.from([
        0x50, 0x4B, 0x05, 0x06,
        0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
        0x00, 0x00,
    ]);
}
class ZipFS extends fslib_1.BasePortableFakeFS {
    constructor(source, opts = {}) {
        super();
        this.listings = new Map();
        this.entries = new Map();
        /**
         * A cache of indices mapped to file sources.
         * Populated by `setFileSource` calls.
         * Required for supporting read after write.
         */
        this.fileSources = new Map();
        this.fds = new Map();
        this.nextFd = 0;
        this.ready = false;
        this.readOnly = false;
        if (opts.readOnly)
            this.readOnly = true;
        const pathOptions = opts;
        this.level = typeof pathOptions.level !== `undefined`
            ? pathOptions.level
            : exports.DEFAULT_COMPRESSION_LEVEL;
        const ZipImplCls = opts.customZipImplementation ?? libzipImpl_1.LibZipImpl;
        if (typeof source === `string`) {
            const { baseFs = new fslib_2.NodeFS() } = pathOptions;
            this.baseFs = baseFs;
            this.path = source;
        }
        else {
            this.path = null;
            this.baseFs = null;
        }
        if (opts.stats) {
            this.stats = opts.stats;
        }
        else {
            if (typeof source === `string`) {
                try {
                    this.stats = this.baseFs.statSync(source);
                }
                catch (error) {
                    if (error.code === `ENOENT` && pathOptions.create) {
                        this.stats = fslib_5.statUtils.makeDefaultStats();
                    }
                    else {
                        throw error;
                    }
                }
            }
            else {
                this.stats = fslib_5.statUtils.makeDefaultStats();
            }
        }
        if (typeof source === `string`) {
            if (opts.create) {
                this.zipImpl = new ZipImplCls({ buffer: makeEmptyArchive(), readOnly: this.readOnly });
            }
            else {
                this.zipImpl = new ZipImplCls({ path: source, baseFs: this.baseFs, readOnly: this.readOnly, size: this.stats.size });
            }
        }
        else {
            this.zipImpl = new ZipImplCls({ buffer: source ?? makeEmptyArchive(), readOnly: this.readOnly });
        }
        this.listings.set(fslib_6.PortablePath.root, new Set());
        const listings = this.zipImpl.getListings();
        for (let t = 0; t < listings.length; t++) {
            const raw = listings[t];
            if (fslib_6.ppath.isAbsolute(raw))
                continue;
            const p = fslib_6.ppath.resolve(fslib_6.PortablePath.root, raw);
            this.registerEntry(p, t);
            // If the raw path is a directory, register it
            // to prevent empty folder being skipped
            if (raw.endsWith(`/`)) {
                this.registerListing(p);
            }
        }
        this.symlinkCount = this.zipImpl.getSymlinkCount();
        this.ready = true;
    }
    getExtractHint(hints) {
        for (const fileName of this.entries.keys()) {
            const ext = this.pathUtils.extname(fileName);
            if (hints.relevantExtensions.has(ext)) {
                return true;
            }
        }
        return false;
    }
    getAllFiles() {
        return Array.from(this.entries.keys());
    }
    getRealPath() {
        if (!this.path)
            throw new Error(`ZipFS don't have real paths when loaded from a buffer`);
        return this.path;
    }
    prepareClose() {
        if (!this.ready)
            throw fslib_5.errors.EBUSY(`archive closed, close`);
        (0, fslib_4.unwatchAllFiles)(this);
    }
    getBufferAndClose() {
        this.prepareClose();
        // zip_source_open on an unlink-after-write empty archive fails with "Entry has been deleted"
        if (this.entries.size === 0) {
            this.discardAndClose();
            return makeEmptyArchive();
        }
        try {
            return this.zipImpl.getBufferAndClose();
        }
        finally {
            this.ready = false;
        }
    }
    discardAndClose() {
        this.prepareClose();
        this.zipImpl.discard();
        this.ready = false;
    }
    saveAndClose() {
        if (!this.path || !this.baseFs)
            throw new Error(`ZipFS cannot be saved and must be discarded when loaded from a buffer`);
        if (this.readOnly) {
            this.discardAndClose();
            return;
        }
        const newMode = this.baseFs.existsSync(this.path) || this.stats.mode === fslib_5.statUtils.DEFAULT_MODE
            ? undefined
            : this.stats.mode;
        this.baseFs.writeFileSync(this.path, this.getBufferAndClose(), { mode: newMode });
        this.ready = false;
    }
    resolve(p) {
        return fslib_6.ppath.resolve(fslib_6.PortablePath.root, p);
    }
    async openPromise(p, flags, mode) {
        return this.openSync(p, flags, mode);
    }
    openSync(p, flags, mode) {
        const fd = this.nextFd++;
        this.fds.set(fd, { cursor: 0, p });
        return fd;
    }
    hasOpenFileHandles() {
        return !!this.fds.size;
    }
    async opendirPromise(p, opts) {
        return this.opendirSync(p, opts);
    }
    opendirSync(p, opts = {}) {
        const resolvedP = this.resolveFilename(`opendir '${p}'`, p);
        if (!this.entries.has(resolvedP) && !this.listings.has(resolvedP))
            throw fslib_5.errors.ENOENT(`opendir '${p}'`);
        const directoryListing = this.listings.get(resolvedP);
        if (!directoryListing)
            throw fslib_5.errors.ENOTDIR(`opendir '${p}'`);
        const entries = [...directoryListing];
        const fd = this.openSync(resolvedP, `r`);
        const onClose = () => {
            this.closeSync(fd);
        };
        return (0, fslib_3.opendir)(this, resolvedP, entries, { onClose });
    }
    async readPromise(fd, buffer, offset, length, position) {
        return this.readSync(fd, buffer, offset, length, position);
    }
    readSync(fd, buffer, offset = 0, length = buffer.byteLength, position = -1) {
        const entry = this.fds.get(fd);
        if (typeof entry === `undefined`)
            throw fslib_5.errors.EBADF(`read`);
        const realPosition = position === -1 || position === null
            ? entry.cursor
            : position;
        const source = this.readFileSync(entry.p);
        source.copy(buffer, offset, realPosition, realPosition + length);
        const bytesRead = Math.max(0, Math.min(source.length - realPosition, length));
        if (position === -1 || position === null)
            entry.cursor += bytesRead;
        return bytesRead;
    }
    async writePromise(fd, buffer, offset, length, position) {
        if (typeof buffer === `string`) {
            return this.writeSync(fd, buffer, position);
        }
        else {
            return this.writeSync(fd, buffer, offset, length, position);
        }
    }
    writeSync(fd, buffer, offset, length, position) {
        const entry = this.fds.get(fd);
        if (typeof entry === `undefined`)
            throw fslib_5.errors.EBADF(`read`);
        throw new Error(`Unimplemented`);
    }
    async closePromise(fd) {
        return this.closeSync(fd);
    }
    closeSync(fd) {
        const entry = this.fds.get(fd);
        if (typeof entry === `undefined`)
            throw fslib_5.errors.EBADF(`read`);
        this.fds.delete(fd);
    }
    createReadStream(p, { encoding } = {}) {
        if (p === null)
            throw new Error(`Unimplemented`);
        const fd = this.openSync(p, `r`);
        const stream = Object.assign(new stream_1.PassThrough({
            emitClose: true,
            autoDestroy: true,
            destroy: (error, callback) => {
                clearImmediate(immediate);
                this.closeSync(fd);
                callback(error);
            },
        }), {
            close() {
                stream.destroy();
            },
            bytesRead: 0,
            path: p,
            // "This property is `true` if the underlying file has not been opened yet"
            pending: false,
        });
        const immediate = setImmediate(async () => {
            try {
                const data = await this.readFilePromise(p, encoding);
                stream.bytesRead = data.length;
                stream.end(data);
            }
            catch (error) {
                stream.destroy(error);
            }
        });
        return stream;
    }
    createWriteStream(p, { encoding } = {}) {
        if (this.readOnly)
            throw fslib_5.errors.EROFS(`open '${p}'`);
        if (p === null)
            throw new Error(`Unimplemented`);
        const chunks = [];
        const fd = this.openSync(p, `w`);
        const stream = Object.assign(new stream_1.PassThrough({
            autoDestroy: true,
            emitClose: true,
            destroy: (error, callback) => {
                try {
                    if (error) {
                        callback(error);
                    }
                    else {
                        this.writeFileSync(p, Buffer.concat(chunks), encoding);
                        callback(null);
                    }
                }
                catch (err) {
                    callback(err);
                }
                finally {
                    this.closeSync(fd);
                }
            },
        }), {
            close() {
                stream.destroy();
            },
            bytesWritten: 0,
            path: p,
            // "This property is `true` if the underlying file has not been opened yet"
            pending: false,
        });
        stream.on(`data`, chunk => {
            const chunkBuffer = Buffer.from(chunk);
            stream.bytesWritten += chunkBuffer.length;
            chunks.push(chunkBuffer);
        });
        return stream;
    }
    async realpathPromise(p) {
        return this.realpathSync(p);
    }
    realpathSync(p) {
        const resolvedP = this.resolveFilename(`lstat '${p}'`, p);
        if (!this.entries.has(resolvedP) && !this.listings.has(resolvedP))
            throw fslib_5.errors.ENOENT(`lstat '${p}'`);
        return resolvedP;
    }
    async existsPromise(p) {
        return this.existsSync(p);
    }
    existsSync(p) {
        if (!this.ready)
            throw fslib_5.errors.EBUSY(`archive closed, existsSync '${p}'`);
        if (this.symlinkCount === 0) {
            const resolvedP = fslib_6.ppath.resolve(fslib_6.PortablePath.root, p);
            return this.entries.has(resolvedP) || this.listings.has(resolvedP);
        }
        let resolvedP;
        try {
            resolvedP = this.resolveFilename(`stat '${p}'`, p, undefined, false);
        }
        catch {
            return false;
        }
        if (resolvedP === undefined)
            return false;
        return this.entries.has(resolvedP) || this.listings.has(resolvedP);
    }
    async accessPromise(p, mode) {
        return this.accessSync(p, mode);
    }
    accessSync(p, mode = fs_1.constants.F_OK) {
        const resolvedP = this.resolveFilename(`access '${p}'`, p);
        if (!this.entries.has(resolvedP) && !this.listings.has(resolvedP))
            throw fslib_5.errors.ENOENT(`access '${p}'`);
        if (this.readOnly && (mode & fs_1.constants.W_OK)) {
            throw fslib_5.errors.EROFS(`access '${p}'`);
        }
    }
    async statPromise(p, opts = { bigint: false }) {
        if (opts.bigint)
            return this.statSync(p, { bigint: true });
        return this.statSync(p);
    }
    statSync(p, opts = { bigint: false, throwIfNoEntry: true }) {
        const resolvedP = this.resolveFilename(`stat '${p}'`, p, undefined, opts.throwIfNoEntry);
        if (resolvedP === undefined)
            return undefined;
        if (!this.entries.has(resolvedP) && !this.listings.has(resolvedP)) {
            if (opts.throwIfNoEntry === false)
                return undefined;
            throw fslib_5.errors.ENOENT(`stat '${p}'`);
        }
        if (p[p.length - 1] === `/` && !this.listings.has(resolvedP))
            throw fslib_5.errors.ENOTDIR(`stat '${p}'`);
        return this.statImpl(`stat '${p}'`, resolvedP, opts);
    }
    async fstatPromise(fd, opts) {
        return this.fstatSync(fd, opts);
    }
    fstatSync(fd, opts) {
        const entry = this.fds.get(fd);
        if (typeof entry === `undefined`)
            throw fslib_5.errors.EBADF(`fstatSync`);
        const { p } = entry;
        const resolvedP = this.resolveFilename(`stat '${p}'`, p);
        if (!this.entries.has(resolvedP) && !this.listings.has(resolvedP))
            throw fslib_5.errors.ENOENT(`stat '${p}'`);
        if (p[p.length - 1] === `/` && !this.listings.has(resolvedP))
            throw fslib_5.errors.ENOTDIR(`stat '${p}'`);
        return this.statImpl(`fstat '${p}'`, resolvedP, opts);
    }
    async lstatPromise(p, opts = { bigint: false }) {
        if (opts.bigint)
            return this.lstatSync(p, { bigint: true });
        return this.lstatSync(p);
    }
    lstatSync(p, opts = { bigint: false, throwIfNoEntry: true }) {
        const resolvedP = this.resolveFilename(`lstat '${p}'`, p, false, opts.throwIfNoEntry);
        if (resolvedP === undefined)
            return undefined;
        if (!this.entries.has(resolvedP) && !this.listings.has(resolvedP)) {
            if (opts.throwIfNoEntry === false)
                return undefined;
            throw fslib_5.errors.ENOENT(`lstat '${p}'`);
        }
        if (p[p.length - 1] === `/` && !this.listings.has(resolvedP))
            throw fslib_5.errors.ENOTDIR(`lstat '${p}'`);
        return this.statImpl(`lstat '${p}'`, resolvedP, opts);
    }
    statImpl(reason, p, opts = {}) {
        const entry = this.entries.get(p);
        // File, or explicit directory
        if (typeof entry !== `undefined`) {
            const stat = this.zipImpl.stat(entry);
            const crc = stat.crc;
            const size = stat.size;
            const mtimeMs = stat.mtime * 1000;
            const uid = this.stats.uid;
            const gid = this.stats.gid;
            const blksize = 512;
            const blocks = Math.ceil(stat.size / blksize);
            const atimeMs = mtimeMs;
            const birthtimeMs = mtimeMs;
            const ctimeMs = mtimeMs;
            const atime = new Date(atimeMs);
            const birthtime = new Date(birthtimeMs);
            const ctime = new Date(ctimeMs);
            const mtime = new Date(mtimeMs);
            const type = this.listings.has(p)
                ? fs_1.constants.S_IFDIR
                : this.isSymbolicLink(entry)
                    ? fs_1.constants.S_IFLNK
                    : fs_1.constants.S_IFREG;
            const defaultMode = type === fs_1.constants.S_IFDIR
                ? 0o755
                : 0o644;
            const mode = type | (this.getUnixMode(entry, defaultMode) & 0o777);
            const statInstance = Object.assign(new fslib_5.statUtils.StatEntry(), { uid, gid, size, blksize, blocks, atime, birthtime, ctime, mtime, atimeMs, birthtimeMs, ctimeMs, mtimeMs, mode, crc });
            return opts.bigint === true ? fslib_5.statUtils.convertToBigIntStats(statInstance) : statInstance;
        }
        // Implicit directory
        if (this.listings.has(p)) {
            const uid = this.stats.uid;
            const gid = this.stats.gid;
            const size = 0;
            const blksize = 512;
            const blocks = 0;
            const atimeMs = this.stats.mtimeMs;
            const birthtimeMs = this.stats.mtimeMs;
            const ctimeMs = this.stats.mtimeMs;
            const mtimeMs = this.stats.mtimeMs;
            const atime = new Date(atimeMs);
            const birthtime = new Date(birthtimeMs);
            const ctime = new Date(ctimeMs);
            const mtime = new Date(mtimeMs);
            const mode = fs_1.constants.S_IFDIR | 0o755;
            const crc = 0;
            const statInstance = Object.assign(new fslib_5.statUtils.StatEntry(), { uid, gid, size, blksize, blocks, atime, birthtime, ctime, mtime, atimeMs, birthtimeMs, ctimeMs, mtimeMs, mode, crc });
            return opts.bigint === true ? fslib_5.statUtils.convertToBigIntStats(statInstance) : statInstance;
        }
        throw new Error(`Unreachable`);
    }
    getUnixMode(index, defaultMode) {
        const [opsys, attributes] = this.zipImpl.getExternalAttributes(index);
        if (opsys !== exports.ZIP_UNIX)
            return defaultMode;
        return attributes >>> 16;
    }
    registerListing(p) {
        const existingListing = this.listings.get(p);
        if (existingListing)
            return existingListing;
        const parentListing = this.registerListing(fslib_6.ppath.dirname(p));
        parentListing.add(fslib_6.ppath.basename(p));
        const newListing = new Set();
        this.listings.set(p, newListing);
        return newListing;
    }
    registerEntry(p, index) {
        const parentListing = this.registerListing(fslib_6.ppath.dirname(p));
        parentListing.add(fslib_6.ppath.basename(p));
        this.entries.set(p, index);
    }
    unregisterListing(p) {
        this.listings.delete(p);
        const parentListing = this.listings.get(fslib_6.ppath.dirname(p));
        parentListing?.delete(fslib_6.ppath.basename(p));
    }
    unregisterEntry(p) {
        this.unregisterListing(p);
        const entry = this.entries.get(p);
        this.entries.delete(p);
        if (typeof entry === `undefined`)
            return;
        this.fileSources.delete(entry);
        if (this.isSymbolicLink(entry)) {
            this.symlinkCount--;
        }
    }
    deleteEntry(p, index) {
        this.unregisterEntry(p);
        this.zipImpl.deleteEntry(index);
    }
    resolveFilename(reason, p, resolveLastComponent = true, throwIfNoEntry = true) {
        if (!this.ready)
            throw fslib_5.errors.EBUSY(`archive closed, ${reason}`);
        let resolvedP = fslib_6.ppath.resolve(fslib_6.PortablePath.root, p);
        if (resolvedP === `/`)
            return fslib_6.PortablePath.root;
        const fileIndex = this.entries.get(resolvedP);
        if (resolveLastComponent && fileIndex !== undefined) {
            if (this.symlinkCount !== 0 && this.isSymbolicLink(fileIndex)) {
                const target = this.getFileSource(fileIndex).toString();
                return this.resolveFilename(reason, fslib_6.ppath.resolve(fslib_6.ppath.dirname(resolvedP), target), true, throwIfNoEntry);
            }
            else {
                return resolvedP;
            }
        }
        while (true) {
            const parentP = this.resolveFilename(reason, fslib_6.ppath.dirname(resolvedP), true, throwIfNoEntry);
            if (parentP === undefined)
                return parentP;
            const isDir = this.listings.has(parentP);
            const doesExist = this.entries.has(parentP);
            if (!isDir && !doesExist) {
                if (throwIfNoEntry === false)
                    return undefined;
                throw fslib_5.errors.ENOENT(reason);
            }
            if (!isDir)
                throw fslib_5.errors.ENOTDIR(reason);
            resolvedP = fslib_6.ppath.resolve(parentP, fslib_6.ppath.basename(resolvedP));
            if (!resolveLastComponent || this.symlinkCount === 0)
                break;
            const index = this.zipImpl.locate(resolvedP.slice(1));
            if (index === -1)
                break;
            if (this.isSymbolicLink(index)) {
                const target = this.getFileSource(index).toString();
                resolvedP = fslib_6.ppath.resolve(fslib_6.ppath.dirname(resolvedP), target);
            }
            else {
                break;
            }
        }
        return resolvedP;
    }
    setFileSource(p, content) {
        const buffer = Buffer.isBuffer(content) ? content : Buffer.from(content);
        const target = fslib_6.ppath.relative(fslib_6.PortablePath.root, p);
        let compression = null;
        if (this.level !== `mixed`) {
            // Use store for level 0, and deflate for 1..9
            const method = this.level === 0
                ? exports.STORE
                : exports.DEFLATE;
            compression = [method, this.level];
        }
        const newIndex = this.zipImpl.setFileSource(target, compression, buffer);
        this.fileSources.set(newIndex, buffer);
        return newIndex;
    }
    isSymbolicLink(index) {
        if (this.symlinkCount === 0)
            return false;
        const [opsys, attrs] = this.zipImpl.getExternalAttributes(index);
        if (opsys !== exports.ZIP_UNIX)
            return false;
        const attributes = attrs >>> 16;
        return (attributes & fs_1.constants.S_IFMT) === fs_1.constants.S_IFLNK;
    }
    getFileSource(index, opts = { asyncDecompress: false }) {
        const cachedFileSource = this.fileSources.get(index);
        if (typeof cachedFileSource !== `undefined`)
            return cachedFileSource;
        const { data, compressionMethod } = this.zipImpl.getFileSource(index);
        if (compressionMethod === exports.STORE) {
            if (this.zipImpl.filesShouldBeCached)
                this.fileSources.set(index, data);
            return data;
        }
        else if (compressionMethod === exports.DEFLATE) {
            if (opts.asyncDecompress) {
                return new Promise((resolve, reject) => {
                    zlib_1.default.inflateRaw(data, (error, result) => {
                        if (error) {
                            reject(error);
                        }
                        else {
                            if (this.zipImpl.filesShouldBeCached)
                                this.fileSources.set(index, result);
                            resolve(result);
                        }
                    });
                });
            }
            else {
                const decompressedData = zlib_1.default.inflateRawSync(data);
                if (this.zipImpl.filesShouldBeCached)
                    this.fileSources.set(index, decompressedData);
                return decompressedData;
            }
        }
        else {
            throw new Error(`Unsupported compression method: ${compressionMethod}`);
        }
    }
    async fchmodPromise(fd, mask) {
        return this.chmodPromise(this.fdToPath(fd, `fchmod`), mask);
    }
    fchmodSync(fd, mask) {
        return this.chmodSync(this.fdToPath(fd, `fchmodSync`), mask);
    }
    async chmodPromise(p, mask) {
        return this.chmodSync(p, mask);
    }
    chmodSync(p, mask) {
        if (this.readOnly)
            throw fslib_5.errors.EROFS(`chmod '${p}'`);
        // We don't allow to make the extracted entries group-writable
        mask &= 0o755;
        const resolvedP = this.resolveFilename(`chmod '${p}'`, p, false);
        const entry = this.entries.get(resolvedP);
        if (typeof entry === `undefined`)
            throw new Error(`Assertion failed: The entry should have been registered (${resolvedP})`);
        const oldMod = this.getUnixMode(entry, fs_1.constants.S_IFREG | 0o000);
        const newMod = oldMod & (~0o777) | mask;
        this.zipImpl.setExternalAttributes(entry, exports.ZIP_UNIX, newMod << 16);
    }
    async fchownPromise(fd, uid, gid) {
        return this.chownPromise(this.fdToPath(fd, `fchown`), uid, gid);
    }
    fchownSync(fd, uid, gid) {
        return this.chownSync(this.fdToPath(fd, `fchownSync`), uid, gid);
    }
    async chownPromise(p, uid, gid) {
        return this.chownSync(p, uid, gid);
    }
    chownSync(p, uid, gid) {
        throw new Error(`Unimplemented`);
    }
    async renamePromise(oldP, newP) {
        return this.renameSync(oldP, newP);
    }
    renameSync(oldP, newP) {
        throw new Error(`Unimplemented`);
    }
    async copyFilePromise(sourceP, destP, flags) {
        const { indexSource, indexDest, resolvedDestP } = this.prepareCopyFile(sourceP, destP, flags);
        const source = await this.getFileSource(indexSource, { asyncDecompress: true });
        const newIndex = this.setFileSource(resolvedDestP, source);
        if (newIndex !== indexDest) {
            this.registerEntry(resolvedDestP, newIndex);
        }
    }
    copyFileSync(sourceP, destP, flags = 0) {
        const { indexSource, indexDest, resolvedDestP } = this.prepareCopyFile(sourceP, destP, flags);
        const source = this.getFileSource(indexSource);
        const newIndex = this.setFileSource(resolvedDestP, source);
        if (newIndex !== indexDest) {
            this.registerEntry(resolvedDestP, newIndex);
        }
    }
    prepareCopyFile(sourceP, destP, flags = 0) {
        if (this.readOnly)
            throw fslib_5.errors.EROFS(`copyfile '${sourceP} -> '${destP}'`);
        if ((flags & fs_1.constants.COPYFILE_FICLONE_FORCE) !== 0)
            throw fslib_5.errors.ENOSYS(`unsupported clone operation`, `copyfile '${sourceP}' -> ${destP}'`);
        const resolvedSourceP = this.resolveFilename(`copyfile '${sourceP} -> ${destP}'`, sourceP);
        const indexSource = this.entries.get(resolvedSourceP);
        if (typeof indexSource === `undefined`)
            throw fslib_5.errors.EINVAL(`copyfile '${sourceP}' -> '${destP}'`);
        const resolvedDestP = this.resolveFilename(`copyfile '${sourceP}' -> ${destP}'`, destP);
        const indexDest = this.entries.get(resolvedDestP);
        if ((flags & (fs_1.constants.COPYFILE_EXCL | fs_1.constants.COPYFILE_FICLONE_FORCE)) !== 0 && typeof indexDest !== `undefined`)
            throw fslib_5.errors.EEXIST(`copyfile '${sourceP}' -> '${destP}'`);
        return {
            indexSource,
            resolvedDestP,
            indexDest,
        };
    }
    async appendFilePromise(p, content, opts) {
        if (this.readOnly)
            throw fslib_5.errors.EROFS(`open '${p}'`);
        if (typeof opts === `undefined`)
            opts = { flag: `a` };
        else if (typeof opts === `string`)
            opts = { flag: `a`, encoding: opts };
        else if (typeof opts.flag === `undefined`)
            opts = { flag: `a`, ...opts };
        return this.writeFilePromise(p, content, opts);
    }
    appendFileSync(p, content, opts = {}) {
        if (this.readOnly)
            throw fslib_5.errors.EROFS(`open '${p}'`);
        if (typeof opts === `undefined`)
            opts = { flag: `a` };
        else if (typeof opts === `string`)
            opts = { flag: `a`, encoding: opts };
        else if (typeof opts.flag === `undefined`)
            opts = { flag: `a`, ...opts };
        return this.writeFileSync(p, content, opts);
    }
    fdToPath(fd, reason) {
        const path = this.fds.get(fd)?.p;
        if (typeof path === `undefined`)
            throw fslib_5.errors.EBADF(reason);
        return path;
    }
    async writeFilePromise(p, content, opts) {
        const { encoding, mode, index, resolvedP } = this.prepareWriteFile(p, opts);
        if (index !== undefined && typeof opts === `object` && opts.flag && opts.flag.includes(`a`))
            content = Buffer.concat([await this.getFileSource(index, { asyncDecompress: true }), Buffer.from(content)]);
        if (encoding !== null)
            // @ts-expect-error: toString ignores unneeded arguments
            content = content.toString(encoding);
        const newIndex = this.setFileSource(resolvedP, content);
        if (newIndex !== index)
            this.registerEntry(resolvedP, newIndex);
        if (mode !== null) {
            await this.chmodPromise(resolvedP, mode);
        }
    }
    writeFileSync(p, content, opts) {
        const { encoding, mode, index, resolvedP } = this.prepareWriteFile(p, opts);
        if (index !== undefined && typeof opts === `object` && opts.flag && opts.flag.includes(`a`))
            content = Buffer.concat([this.getFileSource(index), Buffer.from(content)]);
        if (encoding !== null)
            // @ts-expect-error: toString ignores unneeded arguments
            content = content.toString(encoding);
        const newIndex = this.setFileSource(resolvedP, content);
        if (newIndex !== index)
            this.registerEntry(resolvedP, newIndex);
        if (mode !== null) {
            this.chmodSync(resolvedP, mode);
        }
    }
    prepareWriteFile(p, opts) {
        if (typeof p === `number`)
            p = this.fdToPath(p, `read`);
        if (this.readOnly)
            throw fslib_5.errors.EROFS(`open '${p}'`);
        const resolvedP = this.resolveFilename(`open '${p}'`, p);
        if (this.listings.has(resolvedP))
            throw fslib_5.errors.EISDIR(`open '${p}'`);
        let encoding = null, mode = null;
        if (typeof opts === `string`) {
            encoding = opts;
        }
        else if (typeof opts === `object`) {
            ({
                encoding = null,
                mode = null,
            } = opts);
        }
        const index = this.entries.get(resolvedP);
        return {
            encoding,
            mode,
            resolvedP,
            index,
        };
    }
    async unlinkPromise(p) {
        return this.unlinkSync(p);
    }
    unlinkSync(p) {
        if (this.readOnly)
            throw fslib_5.errors.EROFS(`unlink '${p}'`);
        const resolvedP = this.resolveFilename(`unlink '${p}'`, p);
        if (this.listings.has(resolvedP))
            throw fslib_5.errors.EISDIR(`unlink '${p}'`);
        const index = this.entries.get(resolvedP);
        if (typeof index === `undefined`)
            throw fslib_5.errors.EINVAL(`unlink '${p}'`);
        this.deleteEntry(resolvedP, index);
    }
    async utimesPromise(p, atime, mtime) {
        return this.utimesSync(p, atime, mtime);
    }
    utimesSync(p, atime, mtime) {
        if (this.readOnly)
            throw fslib_5.errors.EROFS(`utimes '${p}'`);
        const resolvedP = this.resolveFilename(`utimes '${p}'`, p);
        this.utimesImpl(resolvedP, mtime);
    }
    async lutimesPromise(p, atime, mtime) {
        return this.lutimesSync(p, atime, mtime);
    }
    lutimesSync(p, atime, mtime) {
        if (this.readOnly)
            throw fslib_5.errors.EROFS(`lutimes '${p}'`);
        const resolvedP = this.resolveFilename(`utimes '${p}'`, p, false);
        this.utimesImpl(resolvedP, mtime);
    }
    utimesImpl(resolvedP, mtime) {
        if (this.listings.has(resolvedP))
            if (!this.entries.has(resolvedP))
                this.hydrateDirectory(resolvedP);
        const entry = this.entries.get(resolvedP);
        if (entry === undefined)
            throw new Error(`Unreachable`);
        this.zipImpl.setMtime(entry, toUnixTimestamp(mtime));
    }
    async mkdirPromise(p, opts) {
        return this.mkdirSync(p, opts);
    }
    mkdirSync(p, { mode = 0o755, recursive = false } = {}) {
        if (recursive)
            return this.mkdirpSync(p, { chmod: mode });
        if (this.readOnly)
            throw fslib_5.errors.EROFS(`mkdir '${p}'`);
        const resolvedP = this.resolveFilename(`mkdir '${p}'`, p);
        if (this.entries.has(resolvedP) || this.listings.has(resolvedP))
            throw fslib_5.errors.EEXIST(`mkdir '${p}'`);
        this.hydrateDirectory(resolvedP);
        this.chmodSync(resolvedP, mode);
        return undefined;
    }
    async rmdirPromise(p, opts) {
        return this.rmdirSync(p, opts);
    }
    rmdirSync(p, { recursive = false } = {}) {
        if (this.readOnly)
            throw fslib_5.errors.EROFS(`rmdir '${p}'`);
        if (recursive) {
            this.removeSync(p);
            return;
        }
        const resolvedP = this.resolveFilename(`rmdir '${p}'`, p);
        const directoryListing = this.listings.get(resolvedP);
        if (!directoryListing)
            throw fslib_5.errors.ENOTDIR(`rmdir '${p}'`);
        if (directoryListing.size > 0)
            throw fslib_5.errors.ENOTEMPTY(`rmdir '${p}'`);
        const index = this.entries.get(resolvedP);
        if (typeof index === `undefined`)
            throw fslib_5.errors.EINVAL(`rmdir '${p}'`);
        this.deleteEntry(p, index);
    }
    async rmPromise(p, opts) {
        return this.rmSync(p, opts);
    }
    rmSync(p, { recursive = false } = {}) {
        if (this.readOnly)
            throw fslib_5.errors.EROFS(`rm '${p}'`);
        if (recursive) {
            this.removeSync(p);
            return;
        }
        const resolvedP = this.resolveFilename(`rm '${p}'`, p);
        const directoryListing = this.listings.get(resolvedP);
        if (!directoryListing)
            throw fslib_5.errors.ENOTDIR(`rm '${p}'`);
        if (directoryListing.size > 0)
            throw fslib_5.errors.ENOTEMPTY(`rm '${p}'`);
        const index = this.entries.get(resolvedP);
        if (typeof index === `undefined`)
            throw fslib_5.errors.EINVAL(`rm '${p}'`);
        this.deleteEntry(p, index);
    }
    hydrateDirectory(resolvedP) {
        const index = this.zipImpl.addDirectory(fslib_6.ppath.relative(fslib_6.PortablePath.root, resolvedP));
        this.registerListing(resolvedP);
        this.registerEntry(resolvedP, index);
        return index;
    }
    async linkPromise(existingP, newP) {
        return this.linkSync(existingP, newP);
    }
    linkSync(existingP, newP) {
        // Zip archives don't support hard links:
        // https://stackoverflow.com/questions/8859616/are-hard-links-possible-within-a-zip-archive
        throw fslib_5.errors.EOPNOTSUPP(`link '${existingP}' -> '${newP}'`);
    }
    async symlinkPromise(target, p) {
        return this.symlinkSync(target, p);
    }
    symlinkSync(target, p) {
        if (this.readOnly)
            throw fslib_5.errors.EROFS(`symlink '${target}' -> '${p}'`);
        const resolvedP = this.resolveFilename(`symlink '${target}' -> '${p}'`, p);
        if (this.listings.has(resolvedP))
            throw fslib_5.errors.EISDIR(`symlink '${target}' -> '${p}'`);
        if (this.entries.has(resolvedP))
            throw fslib_5.errors.EEXIST(`symlink '${target}' -> '${p}'`);
        const index = this.setFileSource(resolvedP, target);
        this.registerEntry(resolvedP, index);
        this.zipImpl.setExternalAttributes(index, exports.ZIP_UNIX, (fs_1.constants.S_IFLNK | 0o777) << 16);
        this.symlinkCount += 1;
    }
    async readFilePromise(p, encoding) {
        // This is messed up regarding the TS signatures
        if (typeof encoding === `object`)
            // @ts-expect-error - reason TBS
            encoding = encoding ? encoding.encoding : undefined;
        const data = await this.readFileBuffer(p, { asyncDecompress: true });
        return encoding ? data.toString(encoding) : data;
    }
    readFileSync(p, encoding) {
        // This is messed up regarding the TS signatures
        if (typeof encoding === `object`)
            // @ts-expect-error - reason TBS
            encoding = encoding ? encoding.encoding : undefined;
        const data = this.readFileBuffer(p);
        return encoding ? data.toString(encoding) : data;
    }
    readFileBuffer(p, opts = { asyncDecompress: false }) {
        if (typeof p === `number`)
            p = this.fdToPath(p, `read`);
        const resolvedP = this.resolveFilename(`open '${p}'`, p);
        if (!this.entries.has(resolvedP) && !this.listings.has(resolvedP))
            throw fslib_5.errors.ENOENT(`open '${p}'`);
        // Ensures that the last component is a directory, if the user said so (even if it is we'll throw right after with EISDIR anyway)
        if (p[p.length - 1] === `/` && !this.listings.has(resolvedP))
            throw fslib_5.errors.ENOTDIR(`open '${p}'`);
        if (this.listings.has(resolvedP))
            throw fslib_5.errors.EISDIR(`read`);
        const entry = this.entries.get(resolvedP);
        if (entry === undefined)
            throw new Error(`Unreachable`);
        return this.getFileSource(entry, opts);
    }
    async readdirPromise(p, opts) {
        return this.readdirSync(p, opts);
    }
    readdirSync(p, opts) {
        const resolvedP = this.resolveFilename(`scandir '${p}'`, p);
        if (!this.entries.has(resolvedP) && !this.listings.has(resolvedP))
            throw fslib_5.errors.ENOENT(`scandir '${p}'`);
        const directoryListing = this.listings.get(resolvedP);
        if (!directoryListing)
            throw fslib_5.errors.ENOTDIR(`scandir '${p}'`);
        if (opts?.recursive) {
            if (opts?.withFileTypes) {
                const entries = Array.from(directoryListing, name => {
                    return Object.assign(this.statImpl(`lstat`, fslib_6.ppath.join(p, name)), {
                        name,
                        path: fslib_6.PortablePath.dot,
                    });
                });
                for (const entry of entries) {
                    if (!entry.isDirectory())
                        continue;
                    const subPath = fslib_6.ppath.join(entry.path, entry.name);
                    const subListing = this.listings.get(fslib_6.ppath.join(resolvedP, subPath));
                    for (const child of subListing) {
                        entries.push(Object.assign(this.statImpl(`lstat`, fslib_6.ppath.join(p, subPath, child)), {
                            name: child,
                            path: subPath,
                        }));
                    }
                }
                return entries;
            }
            else {
                const entries = [...directoryListing];
                for (const subPath of entries) {
                    const subListing = this.listings.get(fslib_6.ppath.join(resolvedP, subPath));
                    if (typeof subListing === `undefined`)
                        continue;
                    for (const child of subListing) {
                        entries.push(fslib_6.ppath.join(subPath, child));
                    }
                }
                return entries;
            }
        }
        else if (opts?.withFileTypes) {
            return Array.from(directoryListing, name => {
                return Object.assign(this.statImpl(`lstat`, fslib_6.ppath.join(p, name)), {
                    name,
                    path: undefined,
                });
            });
        }
        else {
            return [...directoryListing];
        }
    }
    async readlinkPromise(p) {
        const entry = this.prepareReadlink(p);
        return (await this.getFileSource(entry, { asyncDecompress: true })).toString();
    }
    readlinkSync(p) {
        const entry = this.prepareReadlink(p);
        return this.getFileSource(entry).toString();
    }
    prepareReadlink(p) {
        const resolvedP = this.resolveFilename(`readlink '${p}'`, p, false);
        if (!this.entries.has(resolvedP) && !this.listings.has(resolvedP))
            throw fslib_5.errors.ENOENT(`readlink '${p}'`);
        // Ensure that the last component is a directory (if it is we'll throw right after with EISDIR anyway)
        if (p[p.length - 1] === `/` && !this.listings.has(resolvedP))
            throw fslib_5.errors.ENOTDIR(`open '${p}'`);
        if (this.listings.has(resolvedP))
            throw fslib_5.errors.EINVAL(`readlink '${p}'`);
        const entry = this.entries.get(resolvedP);
        if (entry === undefined)
            throw new Error(`Unreachable`);
        if (!this.isSymbolicLink(entry))
            throw fslib_5.errors.EINVAL(`readlink '${p}'`);
        return entry;
    }
    async truncatePromise(p, len = 0) {
        const resolvedP = this.resolveFilename(`open '${p}'`, p);
        const index = this.entries.get(resolvedP);
        if (typeof index === `undefined`)
            throw fslib_5.errors.EINVAL(`open '${p}'`);
        const source = await this.getFileSource(index, { asyncDecompress: true });
        const truncated = Buffer.alloc(len, 0x00);
        source.copy(truncated);
        return await this.writeFilePromise(p, truncated);
    }
    truncateSync(p, len = 0) {
        const resolvedP = this.resolveFilename(`open '${p}'`, p);
        const index = this.entries.get(resolvedP);
        if (typeof index === `undefined`)
            throw fslib_5.errors.EINVAL(`open '${p}'`);
        const source = this.getFileSource(index);
        const truncated = Buffer.alloc(len, 0x00);
        source.copy(truncated);
        return this.writeFileSync(p, truncated);
    }
    async ftruncatePromise(fd, len) {
        return this.truncatePromise(this.fdToPath(fd, `ftruncate`), len);
    }
    ftruncateSync(fd, len) {
        return this.truncateSync(this.fdToPath(fd, `ftruncateSync`), len);
    }
    watch(p, a, b) {
        let persistent;
        switch (typeof a) {
            case `function`:
            case `string`:
            case `undefined`:
                {
                    persistent = true;
                }
                break;
            default:
                {
                    ({ persistent = true } = a);
                }
                break;
        }
        if (!persistent)
            return { on: () => { }, close: () => { } };
        const interval = setInterval(() => { }, 24 * 60 * 60 * 1000);
        return {
            on: () => { },
            close: () => {
                clearInterval(interval);
            },
        };
    }
    watchFile(p, a, b) {
        const resolvedP = fslib_6.ppath.resolve(fslib_6.PortablePath.root, p);
        return (0, fslib_4.watchFile)(this, resolvedP, a, b);
    }
    unwatchFile(p, cb) {
        const resolvedP = fslib_6.ppath.resolve(fslib_6.PortablePath.root, p);
        return (0, fslib_4.unwatchFile)(this, resolvedP, cb);
    }
}
exports.ZipFS = ZipFS;
