"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.openUrl = exports.major = void 0;
exports.getArchitecture = getArchitecture;
exports.getArchitectureName = getArchitectureName;
exports.getArchitectureSet = getArchitectureSet;
exports.getCaller = getCaller;
exports.availableParallelism = availableParallelism;
const tslib_1 = require("tslib");
const fslib_1 = require("@yarnpkg/fslib");
const os_1 = tslib_1.__importDefault(require("os"));
const execUtils = tslib_1.__importStar(require("./execUtils"));
const miscUtils = tslib_1.__importStar(require("./miscUtils"));
exports.major = Number(process.versions.node.split(`.`)[0]);
const openUrlBinary = new Map([
    [`darwin`, `open`],
    [`linux`, `xdg-open`],
    [`win32`, `explorer.exe`],
]).get(process.platform);
exports.openUrl = typeof openUrlBinary !== `undefined`
    ? async (url) => {
        try {
            await execUtils.execvp(openUrlBinary, [url], { cwd: fslib_1.ppath.cwd() });
            return true;
        }
        catch {
            return false;
        }
    }
    : undefined;
const LDD_PATH = `/usr/bin/ldd`;
function getLibc() {
    // Darwin and Windows have their own standard libraries, and the getReport() call is costly.
    // It also seems that Node randomly crashes with no output under some circumstances when running a getReport() on Windows.
    if (process.platform === `darwin` || process.platform === `win32`)
        return null;
    let header;
    try {
        header = fslib_1.xfs.readFileSync(LDD_PATH);
    }
    catch { }
    // Since the getReport can be prohibitely expensive (it also queries DNS which, if misconfigured, can take a long time to timeout),
    // we first check if the ldd binary is glibc or musl, and only then run the getReport() if we can't determine the libc variant.
    if (typeof header !== `undefined`) {
        if (header && (header.includes(`GLIBC`) || header.includes(`libc`)))
            return `glibc`;
        if (header && header.includes(`musl`)) {
            return `musl`;
        }
    }
    const report = process.report?.getReport() ?? {};
    const sharedObjects = report.sharedObjects ?? [];
    // Matches the first group if libc, second group if musl
    const libcRegExp = /\/(?:(ld-linux-|[^/]+-linux-gnu\/)|(libc.musl-|ld-musl-))/;
    return miscUtils.mapAndFind(sharedObjects, entry => {
        const match = entry.match(libcRegExp);
        if (!match)
            return miscUtils.mapAndFind.skip;
        if (match[1])
            return `glibc`;
        if (match[2])
            return `musl`;
        throw new Error(`Assertion failed: Expected the libc variant to have been detected`);
    }) ?? null;
}
let architecture;
let architectureSet;
function getArchitecture() {
    return architecture = architecture ?? {
        os: process.platform,
        cpu: process.arch,
        libc: getLibc(),
    };
}
function getArchitectureName(architecture = getArchitecture()) {
    if (architecture.libc) {
        return `${architecture.os}-${architecture.cpu}-${architecture.libc}`;
    }
    else {
        return `${architecture.os}-${architecture.cpu}`;
    }
}
function getArchitectureSet() {
    const architecture = getArchitecture();
    return architectureSet = architectureSet ?? {
        os: [architecture.os],
        cpu: [architecture.cpu],
        libc: architecture.libc ? [architecture.libc] : [],
    };
}
const chromeRe = /^\s*at (.*?) ?\(((?:file|https?|blob|chrome-extension|native|eval|webpack|<anonymous>|\/|[a-z]:\\|\\\\).*?)(?::(\d+))?(?::(\d+))?\)?\s*$/i;
const chromeEvalRe = /\((\S*)(?::(\d+))(?::(\d+))\)/;
// https://github.com/errwischt/stacktrace-parser/blob/f70768a12579de3469f3fdfdc423657ee6609c7c/src/stack-trace-parser.js
function parseStackLine(line) {
    const parts = chromeRe.exec(line);
    if (!parts)
        return null;
    const isNative = parts[2] && parts[2].indexOf(`native`) === 0; // start of line
    const isEval = parts[2] && parts[2].indexOf(`eval`) === 0; // start of line
    const submatch = chromeEvalRe.exec(parts[2]);
    if (isEval && submatch != null) {
        // throw out eval line/column and use top-most line/column number
        parts[2] = submatch[1]; // url
        parts[3] = submatch[2]; // line
        parts[4] = submatch[3]; // column
    }
    return {
        file: !isNative ? parts[2] : null,
        methodName: parts[1] || `<unknown>`,
        arguments: isNative ? [parts[2]] : [],
        line: parts[3] ? +parts[3] : null,
        column: parts[4] ? +parts[4] : null,
    };
}
function getCaller() {
    const err = new Error();
    const line = err.stack.split(`\n`)[3];
    return parseStackLine(line);
}
function availableParallelism() {
    // TODO: Use os.availableParallelism directly when dropping support for Node.js < 19.4.0
    if (typeof os_1.default.availableParallelism !== `undefined`)
        return os_1.default.availableParallelism();
    return Math.max(1, os_1.default.cpus().length);
}
