"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workspace = void 0;
const tslib_1 = require("tslib");
const fslib_1 = require("@yarnpkg/fslib");
const fast_glob_1 = tslib_1.__importDefault(require("fast-glob"));
const Manifest_1 = require("./Manifest");
const WorkspaceResolver_1 = require("./WorkspaceResolver");
const formatUtils = tslib_1.__importStar(require("./formatUtils"));
const hashUtils = tslib_1.__importStar(require("./hashUtils"));
const semverUtils = tslib_1.__importStar(require("./semverUtils"));
const structUtils = tslib_1.__importStar(require("./structUtils"));
class Workspace {
    constructor(workspaceCwd, { project }) {
        this.workspacesCwds = new Set();
        this.project = project;
        this.cwd = workspaceCwd;
    }
    async setup() {
        this.manifest = await Manifest_1.Manifest.tryFind(this.cwd) ?? new Manifest_1.Manifest();
        // We use ppath.relative to guarantee that the default hash will be consistent even if the project is installed on different OS / path
        // @ts-expect-error: It's ok to initialize it now, even if it's readonly (setup is called right after construction)
        this.relativeCwd = fslib_1.ppath.relative(this.project.cwd, this.cwd) || fslib_1.PortablePath.dot;
        const ident = this.manifest.name ? this.manifest.name : structUtils.makeIdent(null, `${this.computeCandidateName()}-${hashUtils.makeHash(this.relativeCwd).substring(0, 6)}`);
        // @ts-expect-error: It's ok to initialize it now, even if it's readonly (setup is called right after construction)
        this.anchoredDescriptor = structUtils.makeDescriptor(ident, `${WorkspaceResolver_1.WorkspaceResolver.protocol}${this.relativeCwd}`);
        // @ts-expect-error: It's ok to initialize it now, even if it's readonly (setup is called right after construction)
        this.anchoredLocator = structUtils.makeLocator(ident, `${WorkspaceResolver_1.WorkspaceResolver.protocol}${this.relativeCwd}`);
        const patterns = this.manifest.workspaceDefinitions.map(({ pattern }) => pattern);
        if (patterns.length === 0)
            return;
        const relativeCwds = await (0, fast_glob_1.default)(patterns, {
            cwd: fslib_1.npath.fromPortablePath(this.cwd),
            onlyDirectories: true,
            ignore: [`**/node_modules`, `**/.git`, `**/.yarn`],
        });
        // fast-glob returns results in arbitrary order
        relativeCwds.sort();
        await relativeCwds.reduce(async (previousTask, relativeCwd) => {
            const candidateCwd = fslib_1.ppath.resolve(this.cwd, fslib_1.npath.toPortablePath(relativeCwd));
            const exists = await fslib_1.xfs.existsPromise(fslib_1.ppath.join(candidateCwd, `package.json`));
            // Ensure candidateCwds are added in order
            await previousTask;
            if (exists) {
                this.workspacesCwds.add(candidateCwd);
            }
        }, Promise.resolve());
    }
    get anchoredPackage() {
        const pkg = this.project.storedPackages.get(this.anchoredLocator.locatorHash);
        if (!pkg)
            throw new Error(`Assertion failed: Expected workspace ${structUtils.prettyWorkspace(this.project.configuration, this)} (${formatUtils.pretty(this.project.configuration, fslib_1.ppath.join(this.cwd, fslib_1.Filename.manifest), formatUtils.Type.PATH)}) to have been resolved. Run "yarn install" to update the lockfile`);
        return pkg;
    }
    accepts(range) {
        const protocolIndex = range.indexOf(`:`);
        const protocol = protocolIndex !== -1
            ? range.slice(0, protocolIndex + 1)
            : null;
        const pathname = protocolIndex !== -1
            ? range.slice(protocolIndex + 1)
            : range;
        if (protocol === WorkspaceResolver_1.WorkspaceResolver.protocol && fslib_1.ppath.normalize(pathname) === this.relativeCwd)
            return true;
        if (protocol === WorkspaceResolver_1.WorkspaceResolver.protocol && (pathname === `*` || pathname === `^` || pathname === `~`))
            return true;
        const semverRange = semverUtils.validRange(pathname);
        if (!semverRange)
            return false;
        if (protocol === WorkspaceResolver_1.WorkspaceResolver.protocol)
            return semverRange.test(this.manifest.version ?? `0.0.0`);
        if (!this.project.configuration.get(`enableTransparentWorkspaces`))
            return false;
        if (this.manifest.version !== null)
            return semverRange.test(this.manifest.version);
        return false;
    }
    computeCandidateName() {
        if (this.cwd === this.project.cwd) {
            return `root-workspace`;
        }
        else {
            return `${fslib_1.ppath.basename(this.cwd)}` || `unnamed-workspace`;
        }
    }
    /**
     * Find workspaces marked as dependencies/devDependencies of the current workspace recursively.
     *
     * @param rootWorkspace root workspace
     * @param project project
     *
     * @returns all the workspaces marked as dependencies
     */
    getRecursiveWorkspaceDependencies({ dependencies = Manifest_1.Manifest.hardDependencies } = {}) {
        const workspaceList = new Set();
        const visitWorkspace = (workspace) => {
            for (const dependencyType of dependencies) {
                // Quick note: it means that if we have, say, a workspace in
                // dev dependencies but not in dependencies, this workspace will be
                // traversed (even if dependencies traditionally override dev
                // dependencies). It's not clear which behaviour is better, but
                // at least it's consistent.
                for (const descriptor of workspace.manifest[dependencyType].values()) {
                    const foundWorkspace = this.project.tryWorkspaceByDescriptor(descriptor);
                    if (foundWorkspace === null || workspaceList.has(foundWorkspace))
                        continue;
                    workspaceList.add(foundWorkspace);
                    visitWorkspace(foundWorkspace);
                }
            }
        };
        visitWorkspace(this);
        return workspaceList;
    }
    /**
     * Find workspaces which include the current workspace as a dependency/devDependency recursively.
     *
     * @param rootWorkspace root workspace
     * @param project project
     *
     * @returns all the workspaces marked as dependents
     */
    getRecursiveWorkspaceDependents({ dependencies = Manifest_1.Manifest.hardDependencies } = {}) {
        const workspaceList = new Set();
        const visitWorkspace = (workspace) => {
            for (const projectWorkspace of this.project.workspaces) {
                const isDependent = dependencies.some(dependencyType => {
                    return [...projectWorkspace.manifest[dependencyType].values()].some(descriptor => {
                        const foundWorkspace = this.project.tryWorkspaceByDescriptor(descriptor);
                        return foundWorkspace !== null && structUtils.areLocatorsEqual(foundWorkspace.anchoredLocator, workspace.anchoredLocator);
                    });
                });
                if (isDependent && !workspaceList.has(projectWorkspace)) {
                    workspaceList.add(projectWorkspace);
                    visitWorkspace(projectWorkspace);
                }
            }
        };
        visitWorkspace(this);
        return workspaceList;
    }
    /**
     * Retrieves all the child workspaces of a given root workspace recursively
     *
     * @param rootWorkspace root workspace
     * @param project project
     *
     * @returns all the child workspaces
     */
    getRecursiveWorkspaceChildren() {
        const workspaceSet = new Set([this]);
        for (const workspace of workspaceSet) {
            for (const childWorkspaceCwd of workspace.workspacesCwds) {
                const childWorkspace = this.project.workspacesByCwd.get(childWorkspaceCwd);
                if (childWorkspace) {
                    workspaceSet.add(childWorkspace);
                }
            }
        }
        workspaceSet.delete(this);
        return Array.from(workspaceSet);
    }
    async persistManifest() {
        const data = {};
        this.manifest.exportTo(data);
        const path = fslib_1.ppath.join(this.cwd, Manifest_1.Manifest.fileName);
        const content = `${JSON.stringify(data, null, this.manifest.indent)}\n`;
        await fslib_1.xfs.changeFilePromise(path, content, {
            automaticNewlines: true,
        });
        this.manifest.raw = data;
    }
}
exports.Workspace = Workspace;
