"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TelemetryManager = exports.MetricName = void 0;
exports.derive = derive;
const tslib_1 = require("tslib");
const fslib_1 = require("@yarnpkg/fslib");
const YarnVersion_1 = require("./YarnVersion");
const hashUtils = tslib_1.__importStar(require("./hashUtils"));
const httpUtils = tslib_1.__importStar(require("./httpUtils"));
const miscUtils = tslib_1.__importStar(require("./miscUtils"));
const semverUtils = tslib_1.__importStar(require("./semverUtils"));
var MetricName;
(function (MetricName) {
    MetricName["VERSION"] = "version";
    MetricName["COMMAND_NAME"] = "commandName";
    MetricName["PLUGIN_NAME"] = "pluginName";
    MetricName["INSTALL_COUNT"] = "installCount";
    MetricName["PROJECT_COUNT"] = "projectCount";
    MetricName["WORKSPACE_COUNT"] = "workspaceCount";
    MetricName["DEPENDENCY_COUNT"] = "dependencyCount";
    MetricName["EXTENSION"] = "packageExtension";
})(MetricName || (exports.MetricName = MetricName = {}));
function derive(params) {
    const hour = 60 * 60 * 1000;
    const day = 24 * hour;
    const nowDay = Math.floor(params.timeNow / day);
    const updateIntervalMs = params.updateInterval * day;
    const lastUpdate = params.state.lastUpdate ?? params.timeNow + updateIntervalMs + Math.floor(updateIntervalMs * params.randomInitialInterval);
    const nextUpdate = lastUpdate + updateIntervalMs;
    // We reset the tips each day at 8am
    const lastTips = params.state.lastTips ?? nowDay * day;
    const nextTips = lastTips + day + 8 * hour - params.timeZone;
    const triggerUpdate = nextUpdate <= params.timeNow;
    const triggerTips = nextTips <= params.timeNow;
    let nextState = null;
    if (triggerUpdate || triggerTips || !params.state.lastUpdate || !params.state.lastTips) {
        nextState = {};
        nextState.lastUpdate = triggerUpdate ? params.timeNow : lastUpdate;
        nextState.lastTips = lastTips;
        nextState.blocks = triggerUpdate ? {} : params.state.blocks;
        nextState.displayedTips = params.state.displayedTips;
    }
    return { nextState, triggerUpdate, triggerTips, nextTips: triggerTips ? nowDay * day : lastTips };
}
class TelemetryManager {
    constructor(configuration, accountId) {
        this.values = new Map();
        this.hits = new Map();
        this.enumerators = new Map();
        this.nextTips = 0;
        this.displayedTips = [];
        this.shouldCommitTips = false;
        this.configuration = configuration;
        const registryFile = this.getRegistryPath();
        this.isNew = !fslib_1.xfs.existsSync(registryFile);
        this.shouldShowTips = false;
        this.sendReport(accountId);
        this.startBuffer();
    }
    /**
     * Prevents the tip from being displayed today, but doesn't actually display it.
     * We use it when replacing the tip by something else (like an upgrade prompt).
     */
    commitTips() {
        if (this.shouldShowTips) {
            this.shouldCommitTips = true;
        }
    }
    selectTip(allTips) {
        const displayedTips = new Set(this.displayedTips);
        const checkVersion = (selector) => {
            if (selector && YarnVersion_1.YarnVersion) {
                return semverUtils.satisfiesWithPrereleases(YarnVersion_1.YarnVersion, selector);
            }
            else {
                return false;
            }
        };
        // Get all possible non-null messages
        const activeTips = allTips
            .map((_, index) => index)
            .filter(index => allTips[index] && checkVersion(allTips[index]?.selector));
        if (activeTips.length === 0)
            return null;
        // Filter out the ones that have already been displayed
        let availableTips = activeTips
            .filter(index => !displayedTips.has(index));
        // If we've seen all tips, we can reset the list. We still
        // keep the last few items there, just to make sure we don't
        // immediately re-display the same tip as the last past days.
        if (availableTips.length === 0) {
            const sliceLength = Math.floor(activeTips.length * .2);
            this.displayedTips = sliceLength > 0
                ? this.displayedTips.slice(-sliceLength)
                : [];
            availableTips = activeTips
                .filter(index => !displayedTips.has(index));
        }
        const selectedTip = availableTips[Math.floor(Math.random() * availableTips.length)];
        this.displayedTips.push(selectedTip);
        this.commitTips();
        return allTips[selectedTip];
    }
    reportVersion(value) {
        // We don't really care about the exact commit they're using
        this.reportValue(MetricName.VERSION, value.replace(/-git\..*/, `-git`));
    }
    reportCommandName(value) {
        this.reportValue(MetricName.COMMAND_NAME, value || `<none>`);
    }
    reportPluginName(value) {
        this.reportValue(MetricName.PLUGIN_NAME, value);
    }
    reportProject(cwd) {
        this.reportEnumerator(MetricName.PROJECT_COUNT, cwd);
    }
    reportInstall(nodeLinker) {
        this.reportHit(MetricName.INSTALL_COUNT, nodeLinker);
    }
    reportPackageExtension(value) {
        this.reportValue(MetricName.EXTENSION, value);
    }
    reportWorkspaceCount(count) {
        this.reportValue(MetricName.WORKSPACE_COUNT, String(count));
    }
    reportDependencyCount(count) {
        this.reportValue(MetricName.DEPENDENCY_COUNT, String(count));
    }
    reportValue(metric, value) {
        miscUtils.getSetWithDefault(this.values, metric).add(value);
    }
    reportEnumerator(metric, value) {
        miscUtils.getSetWithDefault(this.enumerators, metric).add(hashUtils.makeHash(value));
    }
    reportHit(metric, extra = `*`) {
        const ns = miscUtils.getMapWithDefault(this.hits, metric);
        const current = miscUtils.getFactoryWithDefault(ns, extra, () => 0);
        ns.set(extra, current + 1);
    }
    getRegistryPath() {
        const registryFile = this.configuration.get(`globalFolder`);
        return fslib_1.ppath.join(registryFile, `telemetry.json`);
    }
    sendReport(accountId) {
        const registryFile = this.getRegistryPath();
        let state;
        try {
            state = fslib_1.xfs.readJsonSync(registryFile);
        }
        catch {
            state = {};
        }
        const { nextState, triggerUpdate, triggerTips, nextTips, } = derive({
            state,
            timeNow: Date.now(),
            timeZone: new Date().getTimezoneOffset() * 60 * 1000,
            randomInitialInterval: Math.random(),
            updateInterval: this.configuration.get(`telemetryInterval`),
        });
        this.nextTips = nextTips;
        this.displayedTips = state.displayedTips ?? [];
        if (nextState !== null) {
            try {
                fslib_1.xfs.mkdirSync(fslib_1.ppath.dirname(registryFile), { recursive: true });
                fslib_1.xfs.writeJsonSync(registryFile, nextState);
            }
            catch {
                // In some cases this location is read-only. Too bad 🤷‍♀️
                return false;
            }
        }
        if (triggerTips && this.configuration.get(`enableTips`))
            this.shouldShowTips = true;
        if (triggerUpdate) {
            const blocks = state.blocks ?? {};
            if (Object.keys(blocks).length === 0) {
                const rawUrl = `https://browser-http-intake.logs.datadoghq.eu/v1/input/${accountId}?ddsource=yarn`;
                const sendPayload = (payload) => httpUtils.post(rawUrl, payload, {
                    configuration: this.configuration,
                }).catch(() => {
                    // Nothing we can do
                });
                for (const [userId, block] of Object.entries(state.blocks ?? {})) {
                    if (Object.keys(block).length === 0)
                        continue;
                    const upload = block;
                    upload.userId = userId;
                    upload.reportType = `primary`;
                    for (const key of Object.keys(upload.enumerators ?? {}))
                        upload.enumerators[key] = upload.enumerators[key].length;
                    sendPayload(upload);
                    // Datadog doesn't support well sending multiple tags in a single
                    // payload, so we instead send them separately, at most one value
                    // per query (we still aggregate different tags together).
                    const toSend = new Map();
                    // Also the max amount of queries (at worst once a week, remember)
                    const maxValues = 20;
                    for (const [metricName, values] of Object.entries(upload.values))
                        if (values.length > 0)
                            toSend.set(metricName, values.slice(0, maxValues));
                    while (toSend.size > 0) {
                        const upload = {};
                        upload.userId = userId;
                        upload.reportType = `secondary`;
                        upload.metrics = {};
                        for (const [metricName, values] of toSend) {
                            upload.metrics[metricName] = values.shift();
                            if (values.length === 0) {
                                toSend.delete(metricName);
                            }
                        }
                        sendPayload(upload);
                    }
                }
            }
        }
        return true;
    }
    applyChanges() {
        const registryFile = this.getRegistryPath();
        let state;
        try {
            state = fslib_1.xfs.readJsonSync(registryFile);
        }
        catch {
            state = {};
        }
        const userId = this.configuration.get(`telemetryUserId`) ?? `*`;
        const blocks = state.blocks = state.blocks ?? {};
        const block = blocks[userId] = blocks[userId] ?? {};
        for (const key of this.hits.keys()) {
            const store = block.hits = block.hits ?? {};
            const ns = store[key] = store[key] ?? {};
            for (const [extra, value] of this.hits.get(key)) {
                ns[extra] = (ns[extra] ?? 0) + value;
            }
        }
        for (const field of [`values`, `enumerators`]) {
            for (const key of this[field].keys()) {
                const store = block[field] = block[field] ?? {};
                store[key] = [...new Set([
                        ...store[key] ?? [],
                        ...this[field].get(key) ?? [],
                    ])];
            }
        }
        if (this.shouldCommitTips) {
            state.lastTips = this.nextTips;
            state.displayedTips = this.displayedTips;
        }
        fslib_1.xfs.mkdirSync(fslib_1.ppath.dirname(registryFile), { recursive: true });
        fslib_1.xfs.writeJsonSync(registryFile, state);
    }
    startBuffer() {
        process.on(`exit`, () => {
            try {
                this.applyChanges();
            }
            catch {
                // Explicitly ignore errors
            }
        });
    }
}
exports.TelemetryManager = TelemetryManager;
