"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerPool = exports.AsyncPool = void 0;
const tslib_1 = require("tslib");
const p_limit_1 = tslib_1.__importDefault(require("p-limit"));
const worker_threads_1 = require("worker_threads");
const kTaskInfo = Symbol(`kTaskInfo`);
class AsyncPool {
    constructor(fn, opts) {
        this.fn = fn;
        this.limit = (0, p_limit_1.default)(opts.poolSize);
    }
    run(data) {
        return this.limit(() => this.fn(data));
    }
}
exports.AsyncPool = AsyncPool;
class WorkerPool {
    constructor(source, opts) {
        this.source = source;
        this.workers = [];
        this.limit = (0, p_limit_1.default)(opts.poolSize);
        this.cleanupInterval = setInterval(() => {
            if (this.limit.pendingCount === 0 && this.limit.activeCount === 0) {
                // Start terminating one worker at a time when there are no tasks left.
                // This allows the pool to scale down without having to re-create the
                // entire pool when there is a short amount of time without tasks.
                const worker = this.workers.pop();
                if (worker) {
                    worker.terminate();
                }
                else {
                    clearInterval(this.cleanupInterval);
                }
            }
        }, 5000).unref();
    }
    createWorker() {
        this.cleanupInterval.refresh();
        const worker = new worker_threads_1.Worker(this.source, {
            eval: true,
            execArgv: [...process.execArgv, `--unhandled-rejections=strict`],
        });
        worker.on(`message`, (result) => {
            if (!worker[kTaskInfo])
                throw new Error(`Assertion failed: Worker sent a result without having a task assigned`);
            worker[kTaskInfo].resolve(result);
            worker[kTaskInfo] = null;
            worker.unref();
            this.workers.push(worker);
        });
        worker.on(`error`, err => {
            worker[kTaskInfo]?.reject(err);
            worker[kTaskInfo] = null;
        });
        worker.on(`exit`, code => {
            if (code !== 0)
                worker[kTaskInfo]?.reject(new Error(`Worker exited with code ${code}`));
            worker[kTaskInfo] = null;
        });
        return worker;
    }
    run(data) {
        return this.limit(() => {
            const worker = this.workers.pop() ?? this.createWorker();
            worker.ref();
            return new Promise((resolve, reject) => {
                worker[kTaskInfo] = { resolve, reject };
                worker.postMessage(data);
            });
        });
    }
}
exports.WorkerPool = WorkerPool;
