"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cache = exports.CACHE_VERSION = exports.CACHE_CHECKPOINT = void 0;
const tslib_1 = require("tslib");
const core_1 = require("@yarnpkg/core");
const fslib_1 = require("@yarnpkg/fslib");
const fslib_2 = require("@yarnpkg/fslib");
const libzip_1 = require("@yarnpkg/libzip");
const crypto_1 = require("crypto");
const fs_1 = tslib_1.__importDefault(require("fs"));
const MessageName_1 = require("./MessageName");
const RefCountedCache_1 = require("./RefCountedCache");
const Report_1 = require("./Report");
const hashUtils = tslib_1.__importStar(require("./hashUtils"));
const miscUtils = tslib_1.__importStar(require("./miscUtils"));
const structUtils = tslib_1.__importStar(require("./structUtils"));
/**
 * If value defines the minimal cache version we can read files from. We need
 * to bump this value every time we fix a bug in the cache implementation that
 * causes the archived content to change.
 */
exports.CACHE_CHECKPOINT = miscUtils.parseInt(process.env.YARN_CACHE_CHECKPOINT_OVERRIDE ??
    process.env.YARN_CACHE_VERSION_OVERRIDE ??
    9);
/**
 * The cache version, on the other hand, is meant to be bumped every time we
 * change the archives in any way (for example when upgrading the libzip or zlib
 * implementations in ways that would change the exact bytes). This way we can
 * avoid refetching the archives when their content hasn't actually changed in
 * a significant way.
 */
exports.CACHE_VERSION = miscUtils.parseInt(process.env.YARN_CACHE_VERSION_OVERRIDE ??
    10);
class Cache {
    static async find(configuration, { immutable, check } = {}) {
        const cache = new Cache(configuration.get(`cacheFolder`), { configuration, immutable, check });
        await cache.setup();
        return cache;
    }
    static getCacheKey(configuration) {
        const compressionLevel = configuration.get(`compressionLevel`);
        const cacheSpec = compressionLevel !== `mixed`
            ? `c${compressionLevel}`
            : ``;
        const cacheKey = [
            exports.CACHE_VERSION,
            cacheSpec,
        ].join(``);
        return {
            cacheKey,
            cacheSpec,
        };
    }
    constructor(cacheCwd, { configuration, immutable = configuration.get(`enableImmutableCache`), check = false }) {
        // Contains the list of cache files that got accessed since the last time
        // you cleared the variable. Useful to know which files aren't needed
        // anymore when used in conjunction with fetchEverything.
        this.markedFiles = new Set();
        this.mutexes = new Map();
        this.refCountedZipFsCache = new RefCountedCache_1.RefCountedCache(zipFs => {
            zipFs.discardAndClose();
        });
        /**
         * To ensure different instances of `Cache` doesn't end up copying to the same
         * temporary file this random ID is appended to the filename.
         */
        this.cacheId = `-${(0, crypto_1.randomBytes)(8).toString(`hex`)}.tmp`;
        this.configuration = configuration;
        this.cwd = cacheCwd;
        this.immutable = immutable;
        this.check = check;
        const { cacheSpec, cacheKey, } = Cache.getCacheKey(configuration);
        this.cacheSpec = cacheSpec;
        this.cacheKey = cacheKey;
    }
    get mirrorCwd() {
        if (!this.configuration.get(`enableMirror`))
            return null;
        const mirrorCwd = `${this.configuration.get(`globalFolder`)}/cache`;
        return mirrorCwd !== this.cwd ? mirrorCwd : null;
    }
    getVersionFilename(locator) {
        return `${structUtils.slugifyLocator(locator)}-${this.cacheKey}.zip`;
    }
    getChecksumFilename(locator, checksum) {
        // We only want the actual checksum (not the cache version, since the whole
        // point is to avoid changing the filenames when the cache version changes)
        const contentChecksum = splitChecksumComponents(checksum).hash;
        // We only care about the first few characters. It doesn't matter if that
        // makes the hash easier to collide with, because we check the file hashes
        // during each install anyway.
        const significantChecksum = contentChecksum.slice(0, 10);
        return `${structUtils.slugifyLocator(locator)}-${significantChecksum}.zip`;
    }
    isChecksumCompatible(checksum) {
        // If we don't yet know the checksum, discard the path resolution for now
        // until the checksum can be obtained from somewhere (mirror or network).
        if (checksum === null)
            return false;
        const { cacheVersion, cacheSpec, } = splitChecksumComponents(checksum);
        if (cacheVersion === null)
            return false;
        // The cache keys must always be at least as old as the last checkpoint.
        if (cacheVersion < exports.CACHE_CHECKPOINT)
            return false;
        const migrationMode = this.configuration.get(`cacheMigrationMode`);
        // If the global cache is used, then the lockfile must always be up-to-date,
        // so the archives must be regenerated each time the version changes.
        if (cacheVersion < exports.CACHE_VERSION && migrationMode === `always`)
            return false;
        // If the cache spec changed, we may need to regenerate the archive
        if (cacheSpec !== this.cacheSpec && migrationMode !== `required-only`)
            return false;
        return true;
    }
    getLocatorPath(locator, expectedChecksum) {
        // When using the global cache we want the archives to be named as per
        // the cache key rather than the hash, as otherwise we wouldn't be able
        // to find them if we didn't have the hash (which is the case when adding
        // new dependencies to a project).
        if (this.mirrorCwd === null)
            return fslib_2.ppath.resolve(this.cwd, this.getVersionFilename(locator));
        // Same thing if we don't know the checksum; it means that the package
        // doesn't support being checksum'd (unstablePackage), so we fallback
        // on the versioned filename.
        if (expectedChecksum === null)
            return fslib_2.ppath.resolve(this.cwd, this.getVersionFilename(locator));
        return fslib_2.ppath.resolve(this.cwd, this.getChecksumFilename(locator, expectedChecksum));
    }
    getLocatorMirrorPath(locator) {
        const mirrorCwd = this.mirrorCwd;
        return mirrorCwd !== null ? fslib_2.ppath.resolve(mirrorCwd, this.getVersionFilename(locator)) : null;
    }
    async setup() {
        // mkdir may cause write operations even when directories exist. To ensure that the cache can be successfully used
        // on read-only filesystems, only run mkdir when not running in immutable mode.
        if (!this.configuration.get(`enableGlobalCache`)) {
            if (this.immutable) {
                if (!await fslib_2.xfs.existsPromise(this.cwd)) {
                    throw new Report_1.ReportError(MessageName_1.MessageName.IMMUTABLE_CACHE, `Cache path does not exist.`);
                }
            }
            else {
                await fslib_2.xfs.mkdirPromise(this.cwd, { recursive: true });
                const gitignorePath = fslib_2.ppath.resolve(this.cwd, `.gitignore`);
                await fslib_2.xfs.changeFilePromise(gitignorePath, `/.gitignore\n*.flock\n*.tmp\n`);
            }
        }
        if (this.mirrorCwd || !this.immutable) {
            await fslib_2.xfs.mkdirPromise(this.mirrorCwd || this.cwd, { recursive: true });
        }
    }
    async fetchPackageFromCache(locator, expectedChecksum, { onHit, onMiss, loader, ...opts }) {
        const mirrorPath = this.getLocatorMirrorPath(locator);
        const baseFs = new fslib_1.NodeFS();
        // Conditional packages may not be fetched if they're intended for a
        // different architecture than the current one. To avoid having to be
        // careful about those packages everywhere, we instead change their
        // content to that of an empty in-memory package.
        //
        // This memory representation will be wrapped into an AliasFS to make
        // it seem like it actually exist on the disk, at the location of the
        // cache the package would fill if it was normally fetched.
        const makeMockPackage = () => {
            const zipFs = new libzip_1.ZipFS();
            const rootPackageDir = fslib_2.ppath.join(fslib_1.PortablePath.root, structUtils.getIdentVendorPath(locator));
            zipFs.mkdirSync(rootPackageDir, { recursive: true });
            zipFs.writeJsonSync(fslib_2.ppath.join(rootPackageDir, fslib_1.Filename.manifest), {
                name: structUtils.stringifyIdent(locator),
                mocked: true,
            });
            return zipFs;
        };
        const validateFile = async (path, { isColdHit, controlPath = null }) => {
            // We hide the checksum if the package presence is conditional, because it becomes unreliable
            // so there is no point in computing it unless we're checking the cache
            if (controlPath === null && opts.unstablePackages?.has(locator.locatorHash))
                return { isValid: true, hash: null };
            const actualCacheKey = expectedChecksum && !isColdHit
                ? splitChecksumComponents(expectedChecksum).cacheKey
                : this.cacheKey;
            const actualChecksum = (!opts.skipIntegrityCheck || !expectedChecksum)
                ? `${actualCacheKey}/${await hashUtils.checksumFile(path)}`
                : expectedChecksum;
            if (controlPath !== null) {
                const previousChecksum = (!opts.skipIntegrityCheck || !expectedChecksum)
                    ? `${this.cacheKey}/${await hashUtils.checksumFile(controlPath)}`
                    : expectedChecksum;
                if (actualChecksum !== previousChecksum) {
                    throw new Report_1.ReportError(MessageName_1.MessageName.CACHE_CHECKSUM_MISMATCH, `The remote archive doesn't match the local checksum - has the local cache been corrupted?`);
                }
            }
            let checksumBehavior = null;
            if (expectedChecksum !== null && actualChecksum !== expectedChecksum) {
                // Using --check-cache overrides any preconfigured checksum behavior
                if (this.check) {
                    checksumBehavior = `throw`;
                    // If the lockfile references an old cache format, we tolerate different checksums
                }
                else if (splitChecksumComponents(expectedChecksum).cacheKey !== splitChecksumComponents(actualChecksum).cacheKey) {
                    checksumBehavior = `update`;
                }
                else {
                    checksumBehavior = this.configuration.get(`checksumBehavior`);
                }
            }
            switch (checksumBehavior) {
                case null:
                case `update`:
                    return { isValid: true, hash: actualChecksum };
                case `ignore`:
                    return { isValid: true, hash: expectedChecksum };
                case `reset`:
                    return { isValid: false, hash: expectedChecksum };
                default:
                case `throw`: {
                    throw new Report_1.ReportError(MessageName_1.MessageName.CACHE_CHECKSUM_MISMATCH, `The remote archive doesn't match the expected checksum`);
                }
            }
        };
        const validateFileAgainstRemote = async (cachePath) => {
            if (!loader)
                throw new Error(`Cache check required but no loader configured for ${structUtils.prettyLocator(this.configuration, locator)}`);
            const zipFs = await loader();
            const controlPath = zipFs.getRealPath();
            zipFs.saveAndClose();
            await fslib_2.xfs.chmodPromise(controlPath, 0o644);
            const result = await validateFile(cachePath, {
                controlPath,
                isColdHit: false,
            });
            if (!result.isValid)
                throw new Error(`Assertion failed: Expected a valid checksum`);
            return result.hash;
        };
        const loadPackageThroughMirror = async () => {
            if (mirrorPath === null || !(await fslib_2.xfs.existsPromise(mirrorPath))) {
                const zipFs = await loader();
                const realPath = zipFs.getRealPath();
                zipFs.saveAndClose();
                return { source: `loader`, path: realPath };
            }
            return { source: `mirror`, path: mirrorPath };
        };
        const loadPackage = async () => {
            if (!loader)
                throw new Error(`Cache entry required but missing for ${structUtils.prettyLocator(this.configuration, locator)}`);
            if (this.immutable)
                throw new Report_1.ReportError(MessageName_1.MessageName.IMMUTABLE_CACHE, `Cache entry required but missing for ${structUtils.prettyLocator(this.configuration, locator)}`);
            const { path: packagePath, source: packageSource } = await loadPackageThroughMirror();
            // Do this before moving the file so that we don't pollute the cache with corrupted archives
            const { hash: checksum } = await validateFile(packagePath, {
                isColdHit: true,
            });
            const cachePath = this.getLocatorPath(locator, checksum);
            const copyProcess = [];
            // Copy the package into the mirror
            if (packageSource !== `mirror` && mirrorPath !== null) {
                copyProcess.push(async () => {
                    const mirrorPathTemp = `${mirrorPath}${this.cacheId}`;
                    await fslib_2.xfs.copyFilePromise(packagePath, mirrorPathTemp, fs_1.default.constants.COPYFILE_FICLONE);
                    await fslib_2.xfs.chmodPromise(mirrorPathTemp, 0o644);
                    // Doing a rename is important to ensure the cache is atomic
                    await fslib_2.xfs.renamePromise(mirrorPathTemp, mirrorPath);
                });
            }
            // Copy the package into the cache
            if (!opts.mirrorWriteOnly || mirrorPath === null) {
                copyProcess.push(async () => {
                    const cachePathTemp = `${cachePath}${this.cacheId}`;
                    await fslib_2.xfs.copyFilePromise(packagePath, cachePathTemp, fs_1.default.constants.COPYFILE_FICLONE);
                    await fslib_2.xfs.chmodPromise(cachePathTemp, 0o644);
                    // Doing a rename is important to ensure the cache is atomic
                    await fslib_2.xfs.renamePromise(cachePathTemp, cachePath);
                });
            }
            const finalPath = opts.mirrorWriteOnly
                ? mirrorPath ?? cachePath
                : cachePath;
            await Promise.all(copyProcess.map(copy => copy()));
            return [false, finalPath, checksum];
        };
        const loadPackageThroughMutex = async () => {
            const mutexedLoad = async () => {
                const isUnstablePackage = opts.unstablePackages?.has(locator.locatorHash);
                const tentativeCachePath = isUnstablePackage || !expectedChecksum || this.isChecksumCompatible(expectedChecksum)
                    ? this.getLocatorPath(locator, expectedChecksum)
                    : null;
                const cacheFileExists = tentativeCachePath !== null
                    ? this.markedFiles.has(tentativeCachePath) || await baseFs.existsPromise(tentativeCachePath)
                    : false;
                const shouldMock = !!opts.mockedPackages?.has(locator.locatorHash) && (!this.check || !cacheFileExists);
                const isCacheHit = shouldMock || cacheFileExists;
                const action = isCacheHit
                    ? onHit
                    : onMiss;
                if (action)
                    action();
                if (!isCacheHit) {
                    if (this.immutable && isUnstablePackage)
                        throw new Report_1.ReportError(MessageName_1.MessageName.IMMUTABLE_CACHE, `Cache entry required but missing for ${structUtils.prettyLocator(this.configuration, locator)}; consider defining ${core_1.formatUtils.pretty(this.configuration, `supportedArchitectures`, core_1.formatUtils.Type.CODE)} to cache packages for multiple systems`);
                    return loadPackage();
                }
                else {
                    let checksum = null;
                    const cachePath = tentativeCachePath;
                    if (!shouldMock) {
                        if (this.check) {
                            checksum = await validateFileAgainstRemote(cachePath);
                        }
                        else {
                            const maybeChecksum = await validateFile(cachePath, {
                                isColdHit: false,
                            });
                            if (maybeChecksum.isValid) {
                                checksum = maybeChecksum.hash;
                            }
                            else {
                                return loadPackage();
                            }
                        }
                    }
                    return [shouldMock, cachePath, checksum];
                }
            };
            const mutex = mutexedLoad();
            this.mutexes.set(locator.locatorHash, mutex);
            try {
                return await mutex;
            }
            finally {
                this.mutexes.delete(locator.locatorHash);
            }
        };
        for (let mutex; (mutex = this.mutexes.get(locator.locatorHash));)
            await mutex;
        const [shouldMock, cachePath, checksum] = await loadPackageThroughMutex();
        if (!shouldMock)
            this.markedFiles.add(cachePath);
        const createRefCount = () => this.refCountedZipFsCache.addOrCreate(cachePath, () => {
            return shouldMock
                ? makeMockPackage()
                : new libzip_1.ZipFS(cachePath, { baseFs, readOnly: true });
        });
        let refCountedCacheEntry;
        const lazyFs = new fslib_1.LazyFS(() => miscUtils.prettifySyncErrors(() => {
            refCountedCacheEntry = createRefCount();
            return refCountedCacheEntry.value;
        }, message => {
            return `Failed to open the cache entry for ${structUtils.prettyLocator(this.configuration, locator)}: ${message}`;
        }), fslib_2.ppath);
        // We use an AliasFS to speed up getRealPath calls (e.g. VirtualFetcher.ensureVirtualLink)
        // (there's no need to create the lazy baseFs instance to gather the already-known cachePath)
        const aliasFs = new fslib_1.AliasFS(cachePath, { baseFs: lazyFs, pathUtils: fslib_2.ppath });
        const releaseFs = () => {
            refCountedCacheEntry?.release();
        };
        // We hide the checksum if the package presence is conditional, because it becomes unreliable
        const exposedChecksum = !opts.unstablePackages?.has(locator.locatorHash)
            ? checksum
            : null;
        return [aliasFs, releaseFs, exposedChecksum];
    }
}
exports.Cache = Cache;
const CHECKSUM_REGEX = /^(?:(?<cacheKey>(?<cacheVersion>[0-9]+)(?<cacheSpec>.*))\/)?(?<hash>.*)$/;
function splitChecksumComponents(checksum) {
    const match = checksum.match(CHECKSUM_REGEX);
    if (!match?.groups)
        throw new Error(`Assertion failed: Expected the checksum to match the requested pattern`);
    const cacheVersion = match.groups.cacheVersion
        ? parseInt(match.groups.cacheVersion)
        : null;
    return {
        cacheKey: match.groups.cacheKey ?? null,
        cacheVersion,
        cacheSpec: match.groups.cacheSpec ?? null,
        hash: match.groups.hash,
    };
}
