import { FakeFS, PortablePath, Filename } from '@yarnpkg/fslib';
import { ZipFS } from '@yarnpkg/libzip';
import { Configuration } from './Configuration';
import { LocatorHash, Locator } from './types';
/**
 * If value defines the minimal cache version we can read files from. We need
 * to bump this value every time we fix a bug in the cache implementation that
 * causes the archived content to change.
 */
export declare const CACHE_CHECKPOINT: number;
/**
 * The cache version, on the other hand, is meant to be bumped every time we
 * change the archives in any way (for example when upgrading the libzip or zlib
 * implementations in ways that would change the exact bytes). This way we can
 * avoid refetching the archives when their content hasn't actually changed in
 * a significant way.
 */
export declare const CACHE_VERSION: number;
export type CacheOptions = {
    mockedPackages?: Set<LocatorHash>;
    unstablePackages?: Set<LocatorHash>;
    mirrorWriteOnly?: boolean;
    skipIntegrityCheck?: boolean;
};
export declare class Cache {
    readonly configuration: Configuration;
    readonly cwd: PortablePath;
    readonly markedFiles: Set<PortablePath>;
    readonly immutable: boolean;
    readonly check: boolean;
    readonly cacheKey: string;
    readonly cacheSpec: string;
    private mutexes;
    private refCountedZipFsCache;
    /**
     * To ensure different instances of `Cache` doesn't end up copying to the same
     * temporary file this random ID is appended to the filename.
     */
    private cacheId;
    static find(configuration: Configuration, { immutable, check }?: {
        immutable?: boolean;
        check?: boolean;
    }): Promise<Cache>;
    static getCacheKey(configuration: Configuration): {
        cacheKey: string;
        cacheSpec: string;
    };
    constructor(cacheCwd: PortablePath, { configuration, immutable, check }: {
        configuration: Configuration;
        immutable?: boolean;
        check?: boolean;
    });
    get mirrorCwd(): PortablePath | null;
    getVersionFilename(locator: Locator): Filename;
    getChecksumFilename(locator: Locator, checksum: string): Filename;
    isChecksumCompatible(checksum: string): boolean;
    getLocatorPath(locator: Locator, expectedChecksum: string | null): PortablePath;
    getLocatorMirrorPath(locator: Locator): PortablePath | null;
    setup(): Promise<void>;
    fetchPackageFromCache(locator: Locator, expectedChecksum: string | null, { onHit, onMiss, loader, ...opts }: {
        onHit?: () => void;
        onMiss?: () => void;
        loader?: () => Promise<ZipFS>;
    } & CacheOptions): Promise<[FakeFS<PortablePath>, () => void, string | null]>;
}
