'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var checks = require('@thi.ng/checks');
var api = require('@thi.ng/api');
var hex = require('@thi.ng/hex');

const INV_MAX = 1 / 0xffffffff;
class ARandom {
    float(norm = 1) {
        return this.int() * INV_MAX * norm;
    }
    norm(norm = 1) {
        return (this.int() * INV_MAX - 0.5) * 2 * norm;
    }
    minmax(min, max) {
        return this.float() * (max - min) + min;
    }
}

const random = Math.random;
class SystemRandom extends ARandom {
    int() {
        return (random() * 4294967296)  >>> 0;
    }
    float(norm = 1) {
        return random() * norm;
    }
    norm(norm = 1) {
        return (random() - 0.5) * 2 * norm;
    }
}
const SYSTEM = new SystemRandom();

const randomBytesFrom = (rnd, buf, start = 0, end = buf.length) => {
    for (let i = end; --i >= start;) {
        buf[i] = rnd.int() & 0xff;
    }
    return buf;
};
const randomBytes = checks.hasCrypto()
    ? (buf, start = 0, end = buf.length) => (window.crypto.getRandomValues(buf.subarray(start, end)), buf)
    : (buf, start, end) => randomBytesFrom(SYSTEM, buf, start, end);

class Crypto extends ARandom {
    constructor(size = 64) {
        super();
        this.buffer = new Uint8Array((size + 3) & ~3);
        this.u32 = new Uint32Array(this.buffer.buffer);
        this.i = size >>> 2;
    }
    copy() {
        return new Crypto(this.buffer.length);
    }
    bytes() {
        return new Uint8Array(this.buffer.buffer);
    }
    int() {
        if (this.i >= this.u32.length) {
            randomBytes(this.buffer);
            this.i = 0;
        }
        return this.u32[this.i++];
    }
}
const CRYPTO = new Crypto();

const DEFAULT_SEED_32 = 0xdecafbad;
const DEFAULT_SEED_128 = [
    0xdecafbad,
    0x2fa9d75b,
    0xe41f67e3,
    0x5c83ec1a,
];
const DEFAULT_SEED_160 = [...DEFAULT_SEED_128, 0xf69a5c71];

class Smush32 extends ARandom {
    constructor(seed = DEFAULT_SEED_32) {
        super();
        this.buffer = new Uint32Array([seed, 0]);
    }
    copy() {
        const gen = new Smush32();
        gen.buffer.set(this.buffer);
        return gen;
    }
    seed(s) {
        this.buffer.set([s, 0]);
        return this;
    }
    int() {
        const b = this.buffer;
        const m = 0x5bd1e995;
        const k = (b[1]++ * m) >>> 0;
        const s = (b[0] = ((k ^ (k >> 24) ^ ((b[0] * m) >>> 0)) * m) >>> 0);
        return (s ^ (s >>> 13)) >>> 0;
    }
}

class Xoshiro128 extends ARandom {
    constructor(seed = DEFAULT_SEED_128) {
        super();
        this.buffer = new Uint32Array(4);
        this.seed(seed);
    }
    copy() {
        return new Xoshiro128(this.buffer);
    }
    bytes() {
        return new Uint8Array(this.buffer.buffer);
    }
    seed(seed) {
        this.buffer.set(seed);
        return this;
    }
    int() {
        const s = this.buffer;
        let t = s[0] + s[3];
        const res = ((t << 7) | (t >>> 25)) >>> 0;
        t = s[1] << 9;
        s[2] ^= s[0];
        s[3] ^= s[1];
        s[1] ^= s[2];
        s[0] ^= s[3];
        s[2] ^= t;
        t = s[3];
        s[3] = ((t << 11) | (t >>> 21)) >>> 0;
        return res;
    }
}

class XorShift128 extends ARandom {
    constructor(seed = DEFAULT_SEED_128) {
        super();
        this.buffer = new Uint32Array(4);
        this.seed(seed);
    }
    copy() {
        return new XorShift128(this.buffer);
    }
    bytes() {
        return new Uint8Array(this.buffer.buffer);
    }
    seed(seed) {
        this.buffer.set(seed);
        return this;
    }
    int() {
        const s = this.buffer;
        let t = s[3];
        let w;
        t ^= t << 11;
        t ^= t >>> 8;
        s[3] = s[2];
        s[2] = s[1];
        w = s[1] = s[0];
        return (s[0] = (t ^ w ^ (w >>> 19)) >>> 0);
    }
}

class XorWow extends ARandom {
    constructor(seed = DEFAULT_SEED_160) {
        super();
        this.buffer = new Uint32Array(5);
        this.seed(seed);
    }
    copy() {
        return new XorWow(this.buffer);
    }
    seed(seed) {
        this.buffer.set(seed);
        return this;
    }
    bytes() {
        return new Uint8Array(this.buffer.buffer);
    }
    int() {
        const s = this.buffer;
        let t = s[3];
        let w;
        t ^= t >>> 2;
        t ^= t << 1;
        s[3] = s[2];
        s[2] = s[1];
        w = s[1] = s[0];
        t ^= w;
        t ^= w << 4;
        s[0] = t;
        return (t + (s[4] += 0x587c5)) >>> 0;
    }
}

class XsAdd extends ARandom {
    constructor(seed = DEFAULT_SEED_32) {
        super();
        this.buffer = new Uint32Array(4);
        this.seed(seed);
    }
    bytes() {
        return new Uint8Array(this.buffer.buffer);
    }
    copy() {
        const gen = new XsAdd();
        gen.buffer.set(this.buffer);
        return gen;
    }
    seed(seed) {
        const s = this.buffer;
        s.set([seed, 0, 0, 0]);
        for (let j = 0, i = 1; i < 8; j = i++) {
            let x = (s[j & 3] ^ (s[j & 3] >>> 30)) >>> 0;
            x = (0x8965 * x + (((0x6c07 * x) & 0xffff) << 16)) >>> 0;
            s[i & 3] ^= (i + x) >>> 0;
        }
        return this;
    }
    int() {
        const s = this.buffer;
        let t = s[0];
        t ^= t << 15;
        t ^= t >>> 18;
        t ^= s[3] << 11;
        s[0] = s[1];
        s[1] = s[2];
        s[2] = s[3];
        s[3] = t;
        return (t + s[2]) >>> 0;
    }
}

const coin = (rnd = SYSTEM) => rnd.float() < 0.5;
const fairCoin = (rnd = SYSTEM) => {
    let a, b;
    do {
        a = coin(rnd);
        b = coin(rnd);
    } while (a === b);
    return a;
};

const randomID = (len = 4, prefix = "", syms = "abcdefghijklmnopqrstuvwxyz", rnd = SYSTEM) => {
    const n = syms.length;
    for (; --len >= 0;) {
        prefix += syms[rnd.int() % n];
    }
    return prefix;
};

const uniqueValuesFrom = (k, fn, existing = [], maxTrials = 100) => {
    let n = 0;
    while (n < k) {
        let i;
        let trials = maxTrials;
        do {
            i = fn();
        } while (existing.includes(i) && --trials > 0);
        if (trials <= 0)
            break;
        existing.push(i);
        n++;
    }
    return existing;
};
const uniqueIndices = (k, max, existing, maxTrials = max, rnd = SYSTEM) => {
    api.assert(k >= 0 && k <= max, `k must be in [0, ${max}] interval`);
    return uniqueValuesFrom(k, () => rnd.int() % max, existing, maxTrials);
};

const uuidv4Bytes = (buf, rnd) => {
    buf = buf || new Uint8Array(16);
    buf = rnd ? randomBytesFrom(rnd, buf) : randomBytes(buf);
    buf[6] = 0x40 | (buf[6] & 0x0f);
    buf[8] = 0x80 | (buf[8] & 0x3f);
    return buf;
};
const uuid = (id, i = 0) => hex.uuid(id || uuidv4Bytes(), i);

const weightedRandom = (choices, weights, rnd = SYSTEM) => {
    const n = choices.length;
    api.assert(n > 0, "no choices given");
    const opts = weights
        ? choices
            .map((x, i) => [weights[i] || 0, x])
            .sort((a, b) => b[0] - a[0])
        : choices.map((x) => [1, x]);
    const total = opts.reduce((acc, o) => acc + o[0], 0);
    total <= 0 && console.warn("total weights <= 0");
    return () => {
        const r = rnd.float(total);
        let sum = total;
        for (let i = 0; i < n; i++) {
            sum -= opts[i][0];
            if (sum <= r) {
                return opts[i][1];
            }
        }
        return undefined;
    };
};

const exponential = (rnd = SYSTEM, lambda = 10) => lambda === 0 ? () => Infinity : () => -Math.log(1 - rnd.float(1)) / lambda;

const gaussian = (rnd = SYSTEM, n = 24, offset = 0, scale = 1) => () => {
    let sum = 0;
    let m = n;
    while (m-- > 0)
        sum += rnd.norm(scale);
    return sum / n + offset;
};

const geometric = (rnd = SYSTEM, p = 0.5) => p <= 0
    ? () => Infinity
    : p >= 1
        ? () => 1
        : ((p = Math.log(1 - p)),
            () => Math.floor(Math.log(1 - rnd.float(1)) / p) + 1);

const normal = (rnd = SYSTEM, bias = 0, sigma = 1) => {
    let a;
    let b;
    let r;
    return () => {
        if (a != null) {
            b = a;
            a = null;
        }
        else {
            do {
                a = rnd.norm();
                b = rnd.norm();
                r = a * a + b * b;
            } while (r > 1 || r === 0);
        }
        return bias + sigma * b * Math.sqrt((-2 * Math.log(r)) / r);
    };
};

const uniform = (rnd = SYSTEM, min = 0, max = 1) => () => rnd.minmax(min, max);

exports.ARandom = ARandom;
exports.CRYPTO = CRYPTO;
exports.Crypto = Crypto;
exports.SYSTEM = SYSTEM;
exports.Smush32 = Smush32;
exports.SystemRandom = SystemRandom;
exports.XorShift128 = XorShift128;
exports.XorWow = XorWow;
exports.Xoshiro128 = Xoshiro128;
exports.XsAdd = XsAdd;
exports.coin = coin;
exports.exponential = exponential;
exports.fairCoin = fairCoin;
exports.gaussian = gaussian;
exports.geometric = geometric;
exports.normal = normal;
exports.randomBytes = randomBytes;
exports.randomBytesFrom = randomBytesFrom;
exports.randomID = randomID;
exports.uniform = uniform;
exports.uniqueIndices = uniqueIndices;
exports.uniqueValuesFrom = uniqueValuesFrom;
exports.uuid = uuid;
exports.uuidv4Bytes = uuidv4Bytes;
exports.weightedRandom = weightedRandom;
