'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

const isDeref = (x) => x != null && typeof x["deref"] === "function";
const deref = (x) => (isDeref(x) ? x.deref() : x);

exports.LogLevel = void 0;
(function (LogLevel) {
    LogLevel[LogLevel["FINE"] = 0] = "FINE";
    LogLevel[LogLevel["DEBUG"] = 1] = "DEBUG";
    LogLevel[LogLevel["INFO"] = 2] = "INFO";
    LogLevel[LogLevel["WARN"] = 3] = "WARN";
    LogLevel[LogLevel["SEVERE"] = 4] = "SEVERE";
    LogLevel[LogLevel["NONE"] = 5] = "NONE";
})(exports.LogLevel || (exports.LogLevel = {}));

exports.GLType = void 0;
(function (GLType) {
    GLType[GLType["I8"] = 5120] = "I8";
    GLType[GLType["U8"] = 5121] = "U8";
    GLType[GLType["I16"] = 5122] = "I16";
    GLType[GLType["U16"] = 5123] = "U16";
    GLType[GLType["I32"] = 5124] = "I32";
    GLType[GLType["U32"] = 5125] = "U32";
    GLType[GLType["F32"] = 5126] = "F32";
})(exports.GLType || (exports.GLType = {}));
const GL2TYPE = {
    [exports.GLType.I8]: "i8",
    [exports.GLType.U8]: "u8",
    [exports.GLType.I16]: "i16",
    [exports.GLType.U16]: "u16",
    [exports.GLType.I32]: "i32",
    [exports.GLType.U32]: "u32",
    [exports.GLType.F32]: "f32",
};
const TYPE2GL = {
    i8: exports.GLType.I8,
    u8: exports.GLType.U8,
    u8c: exports.GLType.U8,
    i16: exports.GLType.I16,
    u16: exports.GLType.U16,
    i32: exports.GLType.I32,
    u32: exports.GLType.U32,
    f32: exports.GLType.F32,
    f64: undefined,
};
const SIZEOF = {
    u8: 1,
    u8c: 1,
    i8: 1,
    u16: 2,
    i16: 2,
    u32: 4,
    i32: 4,
    f32: 4,
    f64: 8,
};
const FLOAT_ARRAY_CTORS = {
    f32: Float32Array,
    f64: Float64Array,
};
const INT_ARRAY_CTORS = {
    i8: Int8Array,
    i16: Int16Array,
    i32: Int32Array,
};
const UINT_ARRAY_CTORS = {
    u8: Uint8Array,
    u8c: Uint8ClampedArray,
    u16: Uint16Array,
    u32: Uint32Array,
};
const TYPEDARRAY_CTORS = Object.assign(Object.assign(Object.assign({}, FLOAT_ARRAY_CTORS), INT_ARRAY_CTORS), UINT_ARRAY_CTORS);
const asNativeType = (type) => {
    const t = GL2TYPE[type];
    return t !== undefined ? t : type;
};
const asGLType = (type) => {
    const t = TYPE2GL[type];
    return t !== undefined ? t : type;
};
const sizeOf = (type) => SIZEOF[asNativeType(type)];
function typedArray(type, ...xs) {
    return new TYPEDARRAY_CTORS[asNativeType(type)](...xs);
}
const typedArrayType = (x) => {
    if (Array.isArray(x))
        return "f64";
    for (let id in TYPEDARRAY_CTORS) {
        if (x instanceof TYPEDARRAY_CTORS[id])
            return id;
    }
    return "f64";
};
const uintTypeForSize = (x) => x <= 0x100 ? "u8" : x <= 0x10000 ? "u16" : "u32";
const intTypeForSize = (x) => x >= -0x80 && x < 0x80 ? "i8" : x >= -0x8000 && x < 0x8000 ? "i16" : "i32";
const uintTypeForBits = (x) => x > 16 ? "u32" : x > 8 ? "u16" : "u8";
const intTypeForBits = (x) => x > 16 ? "i32" : x > 8 ? "i16" : "i8";

const DEFAULT_EPS = 1e-6;
const SEMAPHORE = Symbol();
const NO_OP = () => { };
const EVENT_ALL = "*";
const EVENT_ENABLE = "enable";
const EVENT_DISABLE = "disable";

const assert = (() => {
    try {
        return (process.env.NODE_ENV !== "production" ||
            process.env.UMBRELLA_ASSERTS === "1");
    }
    catch (e) { }
    return false;
})()
    ? (test, msg = "assertion failed") => {
        if ((typeof test === "function" && !test()) || !test) {
            throw new Error(typeof msg === "function" ? msg() : msg);
        }
    }
    : NO_OP;

const exposeGlobal = (id, value, always = false) => {
    const glob = typeof global !== "undefined"
        ? global
        : typeof window !== "undefined"
            ? window
            : undefined;
    if (glob &&
        (always ||
            (() => {
                try {
                    return (process.env.NODE_ENV !== "production" ||
                        process.env.UMBRELLA_GLOBALS === "1");
                }
                catch (e) { }
                return false;
            })())) {
        glob[id] = value;
    }
};

const NULL_LOGGER = Object.freeze({
    level: exports.LogLevel.NONE,
    fine() { },
    debug() { },
    info() { },
    warn() { },
    severe() { },
});
class ConsoleLogger {
    constructor(id, level = exports.LogLevel.FINE) {
        this.id = id;
        this.level = level;
    }
    fine(...args) {
        this.level <= exports.LogLevel.FINE && this.log("FINE", args);
    }
    debug(...args) {
        this.level <= exports.LogLevel.DEBUG && this.log("DEBUG", args);
    }
    info(...args) {
        this.level <= exports.LogLevel.INFO && this.log("INFO", args);
    }
    warn(...args) {
        this.level <= exports.LogLevel.WARN && this.log("WARN", args);
    }
    severe(...args) {
        this.level <= exports.LogLevel.SEVERE && this.log("SEVERE", args);
    }
    log(level, args) {
        console.log(`[${level}] ${this.id}:`, ...args);
    }
}

const mixin = (behaviour, sharedBehaviour = {}) => {
    const instanceKeys = Reflect.ownKeys(behaviour);
    const sharedKeys = Reflect.ownKeys(sharedBehaviour);
    const typeTag = Symbol("isa");
    function _mixin(clazz) {
        for (let key of instanceKeys) {
            const existing = Object.getOwnPropertyDescriptor(clazz.prototype, key);
            if (!existing || existing.configurable) {
                Object.defineProperty(clazz.prototype, key, {
                    value: behaviour[key],
                    writable: true,
                });
            }
            else {
                console.log(`not patching: ${clazz.name}.${key.toString()}`);
            }
        }
        Object.defineProperty(clazz.prototype, typeTag, { value: true });
        return clazz;
    }
    for (let key of sharedKeys) {
        Object.defineProperty(_mixin, key, {
            value: sharedBehaviour[key],
            enumerable: sharedBehaviour.propertyIsEnumerable(key),
        });
    }
    Object.defineProperty(_mixin, Symbol.hasInstance, {
        value: (x) => !!x[typeTag],
    });
    return _mixin;
};

const configurable = (state) => function (_, __, descriptor) {
    descriptor.configurable = state;
};

const deprecated = (msg, log = console.log) => function (target, prop, descriptor) {
    const signature = `${target.constructor.name}#${prop.toString()}`;
    const fn = descriptor.value;
    assert(typeof fn === "function", `${signature} is not a function`);
    descriptor.value = function () {
        log(`DEPRECATED ${signature}: ${msg || "will be removed soon"}`);
        return fn.apply(this, arguments);
    };
    return descriptor;
};

const nomixin = (_, __, descriptor) => {
    descriptor.configurable = false;
};

const sealed = (constructor) => {
    Object.seal(constructor);
    Object.seal(constructor.prototype);
};

const IEnableMixin = mixin({
    _enabled: true,
    isEnabled() {
        return this._enabled;
    },
    enable() {
        $enable(this, true, EVENT_ENABLE);
    },
    disable() {
        $enable(this, false, EVENT_DISABLE);
    },
    toggle() {
        this._enabled ? this.disable() : this.enable();
        return this._enabled;
    },
});
const $enable = (target, state, id) => {
    target._enabled = state;
    if (target.notify) {
        target.notify({ id, target });
    }
};

const inotify_dispatch = (listeners, e) => {
    if (!listeners)
        return;
    for (let i = 0, n = listeners.length, l; i < n; i++) {
        l = listeners[i];
        l[0].call(l[1], e);
        if (e.canceled) {
            return;
        }
    }
};
const INotifyMixin = mixin({
    addListener(id, fn, scope) {
        let l = (this._listeners = this._listeners || {})[id];
        !l && (l = this._listeners[id] = []);
        if (this.__listener(l, fn, scope) === -1) {
            l.push([fn, scope]);
            return true;
        }
        return false;
    },
    removeListener(id, fn, scope) {
        let listeners;
        if (!(listeners = this._listeners))
            return false;
        const l = listeners[id];
        if (l) {
            const idx = this.__listener(l, fn, scope);
            if (idx !== -1) {
                l.splice(idx, 1);
                !l.length && delete listeners[id];
                return true;
            }
        }
        return false;
    },
    notify(e) {
        let listeners;
        if (!(listeners = this._listeners))
            return false;
        e.target === undefined && (e.target = this);
        inotify_dispatch(listeners[e.id], e);
        inotify_dispatch(listeners[EVENT_ALL], e);
    },
    __listener(listeners, f, scope) {
        let i = listeners.length;
        while (--i >= 0) {
            const l = listeners[i];
            if (l[0] === f && l[1] === scope) {
                break;
            }
        }
        return i;
    },
});

const iterable = (prop) => mixin({
    *[Symbol.iterator]() {
        yield* this[prop];
    },
});

const IWatchMixin = mixin({
    addWatch(id, fn) {
        this._watches = this._watches || {};
        if (this._watches[id]) {
            return false;
        }
        this._watches[id] = fn;
        return true;
    },
    removeWatch(id) {
        if (!this._watches)
            return;
        if (this._watches[id]) {
            delete this._watches[id];
            return true;
        }
        return false;
    },
    notifyWatches(oldState, newState) {
        if (!this._watches)
            return;
        const w = this._watches;
        for (let id in w) {
            w[id](id, oldState, newState);
        }
    },
});

exports.ConsoleLogger = ConsoleLogger;
exports.DEFAULT_EPS = DEFAULT_EPS;
exports.EVENT_ALL = EVENT_ALL;
exports.EVENT_DISABLE = EVENT_DISABLE;
exports.EVENT_ENABLE = EVENT_ENABLE;
exports.FLOAT_ARRAY_CTORS = FLOAT_ARRAY_CTORS;
exports.GL2TYPE = GL2TYPE;
exports.IEnableMixin = IEnableMixin;
exports.INT_ARRAY_CTORS = INT_ARRAY_CTORS;
exports.INotifyMixin = INotifyMixin;
exports.IWatchMixin = IWatchMixin;
exports.NO_OP = NO_OP;
exports.NULL_LOGGER = NULL_LOGGER;
exports.SEMAPHORE = SEMAPHORE;
exports.SIZEOF = SIZEOF;
exports.TYPE2GL = TYPE2GL;
exports.TYPEDARRAY_CTORS = TYPEDARRAY_CTORS;
exports.UINT_ARRAY_CTORS = UINT_ARRAY_CTORS;
exports.asGLType = asGLType;
exports.asNativeType = asNativeType;
exports.assert = assert;
exports.configurable = configurable;
exports.deprecated = deprecated;
exports.deref = deref;
exports.exposeGlobal = exposeGlobal;
exports.inotify_dispatch = inotify_dispatch;
exports.intTypeForBits = intTypeForBits;
exports.intTypeForSize = intTypeForSize;
exports.isDeref = isDeref;
exports.iterable = iterable;
exports.mixin = mixin;
exports.nomixin = nomixin;
exports.sealed = sealed;
exports.sizeOf = sizeOf;
exports.typedArray = typedArray;
exports.typedArrayType = typedArrayType;
exports.uintTypeForBits = uintTypeForBits;
exports.uintTypeForSize = uintTypeForSize;
