"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.eq = eq;
exports.gt = gt;
exports.gte = gte;
exports.lt = lt;
exports.lte = lte;
exports.valid = valid;
exports.satisfies = satisfies;
exports.maxSatisfying = maxSatisfying;
exports.minSatisfying = minSatisfying;
exports.major = major;
exports.minor = minor;
exports.patch = patch;
exports.prerelease = prerelease;
const version_1 = require("./ruby/version");
const requirement_1 = require("./ruby/requirement");
/**
 * v1 == v2 This is true if they're logically equivalent, even if they're not the exact same string. You already know how to compare strings.
 */
function eq(v1, v2) {
    const x = version_1.Version.create(v1);
    const y = version_1.Version.create(v2);
    return x.compare(y) === 0;
}
/**
 * v1 > v2
 */
function gt(v1, v2) {
    const x = version_1.Version.create(v1);
    const y = version_1.Version.create(v2);
    return x.compare(y) === 1;
}
/**
 * v1 >= v2
 */
function gte(v1, v2) {
    const x = version_1.Version.create(v1);
    const y = version_1.Version.create(v2);
    return x.compare(y) !== -1;
}
/**
 * v1 > v2
 */
function lt(v1, v2) {
    const x = version_1.Version.create(v1);
    const y = version_1.Version.create(v2);
    return x.compare(y) === -1;
}
/**
 * v1 >= v2
 */
function lte(v1, v2) {
    const x = version_1.Version.create(v1);
    const y = version_1.Version.create(v2);
    return x.compare(y) !== 1;
}
/**
 * Return the parsed version, or null if it's not valid.
 */
function valid(version) {
    if (!version)
        return null;
    return version_1.Version.isCorrect(version) ? version : null;
}
/**
 * Return true if the version satisfies the range.
 */
function satisfies(version, range) {
    try {
        const v = new version_1.Version(version);
        const r = new requirement_1.Requirement(...range.split(/\s*,\s*/));
        return r.isSatisfiedBy(v);
    }
    catch {
        return false;
    }
}
/**
 * Return the highest version in the list that satisfies the range, or null if none of them do.
 */
function maxSatisfying(versions, range) {
    return versions.reduce((x, y) => {
        const isValid = satisfies(y, range);
        if (isValid && (!x || lt(x, y)))
            return y;
        return x;
    }, null);
}
/**
 * Return the lowest version in the list that satisfies the range, or null if none of them do.
 */
function minSatisfying(versions, range) {
    return versions.reduce((x, y) => {
        const isValid = satisfies(y, range);
        if (isValid && (!x || gt(x, y)))
            return y;
        return x;
    }, null);
}
/**
 * Return the major version number.
 */
function major(v) {
    if (!v)
        return null;
    const version = version_1.Version.create(v);
    if (!version)
        return null;
    const [segments] = version.splitSegments();
    const [x] = segments;
    return x;
}
/**
 * Return the minor version number.
 */
function minor(v) {
    if (!v)
        return null;
    const version = version_1.Version.create(v);
    if (!version)
        return null;
    const [segments] = version.splitSegments();
    const [, x] = segments;
    return typeof x === 'number' ? x : null;
}
/**
 * Return the patch version number.
 */
function patch(v) {
    if (!v)
        return null;
    const version = version_1.Version.create(v);
    if (!version)
        return null;
    const [segments] = version.splitSegments();
    const [, , x] = segments;
    return typeof x === 'number' ? x : null;
}
/**
 * Returns an array of prerelease components, or null if none exist.
 */
function prerelease(v) {
    if (!v)
        return null;
    const version = version_1.Version.create(v);
    if (!version)
        return null;
    const [, segments] = version.splitSegments();
    return segments.length ? segments.map((x) => x.toString()) : null;
}
