"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveFromCatalog = resolveFromCatalog;
const error_1 = require("@pnpm/error");
const catalogs_protocol_parser_1 = require("@pnpm/catalogs.protocol-parser");
function resolveFromCatalog(catalogs, wantedDependency) {
    const catalogName = (0, catalogs_protocol_parser_1.parseCatalogProtocol)(wantedDependency.bareSpecifier);
    if (catalogName == null) {
        return { type: 'unused' };
    }
    const catalogLookup = catalogs[catalogName]?.[wantedDependency.alias];
    if (catalogLookup == null) {
        return {
            type: 'misconfiguration',
            catalogName,
            error: new error_1.PnpmError('CATALOG_ENTRY_NOT_FOUND_FOR_SPEC', `No catalog entry '${wantedDependency.alias}' was found for catalog '${catalogName}'.`),
        };
    }
    if ((0, catalogs_protocol_parser_1.parseCatalogProtocol)(catalogLookup) != null) {
        return {
            type: 'misconfiguration',
            catalogName,
            error: new error_1.PnpmError('CATALOG_ENTRY_INVALID_RECURSIVE_DEFINITION', `Found invalid catalog entry using the catalog protocol recursively. The entry for '${wantedDependency.alias}' in catalog '${catalogName}' is invalid.`),
        };
    }
    // Ban catalog entries that use the workspace protocol for a few reasons:
    //
    //   1. It's kind of silly. It'd be better to encourage users to use the
    //      workspace protocol directly.
    //   2. Catalogs cache the resolved version of a dependency specifier in
    //      pnpm-lock.yaml for more consistent resolution across importers. The
    //      link: resolutions can't be shared between importers.
    const protocolOfLookup = catalogLookup.split(':')[0];
    if (protocolOfLookup === 'workspace') {
        return {
            type: 'misconfiguration',
            catalogName,
            error: new error_1.PnpmError('CATALOG_ENTRY_INVALID_WORKSPACE_SPEC', `The workspace protocol cannot be used as a catalog value. The entry for '${wantedDependency.alias}' in catalog '${catalogName}' is invalid.`),
        };
    }
    // A future version of pnpm will try to support this. These protocols aren't
    // supported today since these are often relative file paths that users expect
    // to be relative to the repo root rather than the location of the pnpm
    // workspace package.
    if (['link', 'file'].includes(protocolOfLookup)) {
        return {
            type: 'misconfiguration',
            catalogName,
            error: new error_1.PnpmError('CATALOG_ENTRY_INVALID_SPEC', `The entry for '${wantedDependency.alias}' in catalog '${catalogName}' declares a dependency using the '${protocolOfLookup}' protocol. This is not yet supported, but may be in a future version of pnpm.`),
        };
    }
    return {
        type: 'found',
        resolution: {
            catalogName,
            specifier: catalogLookup,
        },
    };
}
//# sourceMappingURL=resolveFromCatalog.js.map