declare const RE_IGNORECASE = 1;
declare const RE_EXTENDED = 2;
declare const RE_MULTILINE = 4;
declare const S_EXTENDS = "__ruby_extends__";
declare const S_DEFAULT = "__ruby_default__";

type Hash = Record<string | number | symbol, unknown>;
declare class RubyBaseObject {
    data?: unknown;
    wrapped?: string | Uint8Array | RegExp | unknown[] | Hash | RubyHash;
    userDefined?: Uint8Array;
    userMarshal?: unknown;
}
declare class RubyObject extends RubyBaseObject {
    class: symbol;
    constructor(class_: symbol);
}
declare class RubyStruct extends RubyBaseObject {
    class: symbol;
    members: Record<symbol, unknown>;
    constructor(class_: symbol, members?: Record<symbol, unknown>);
}
declare class RubyClass extends RubyBaseObject {
    name: string;
    constructor(name: string);
}
declare class RubyModule extends RubyBaseObject {
    name: string;
    old?: boolean | undefined;
    constructor(name: string, old?: boolean | undefined);
}
declare class RubyRegexp {
    source: string;
    options: number;
    constructor(source?: string, options?: number);
}
declare class RubyHash {
    entries: [unknown, unknown][];
    default?: unknown;
    constructor(entries?: [unknown, unknown][], default_?: unknown);
}
declare class RubyRange extends RubyObject {
    constructor(begin: unknown | null, end: unknown | null, exclusive: boolean);
}
interface RubyNumeric {
    readonly isInteger: boolean;
    valueOf(): number;
}
declare class RubyInteger implements RubyNumeric {
    value: number;
    isInteger: true;
    constructor(value: number);
    valueOf(): number;
}
declare class RubyFloat implements RubyNumeric {
    value: number;
    isInteger: false;
    constructor(value: number);
    valueOf(): number;
}

interface ClassLike {
    readonly name: string;
    readonly prototype: object | null;
}
interface LoadOptions {
    /**
     * If set, force the encoding of strings, otherwise strings will be decoded on demand.
     * ```js
     * load(data) // => ["foo", Uint8Array(3) [102, 111, 111]]
     * load(data, { string: "utf8" }) // => ["foo", "foo"]
     * load(data, { string: "binary" }) // => [Uint8Array(3) [102, 111, 111], Uint8Array(3) [102, 111, 111]]
     * ```
     */
    string?: "binary" | "utf8";
    /**
     * If set, put integers and floats in RubyInteger and RubyFloat.
     * No bigint support now.
     * ```js
     * load(data) // => 1
     * load(data, { numeric: "wrap" }) // => RubyFloat { value: 1 }
     * load(data, { numeric: "wrap" }) // => RubyInteger { value: 1 }
     * ```
     */
    numeric?: "wrap";
    /**
     * If true, convert symbol keys to string when decoding ruby Hash in JS objects.
     * ```js
     * load(data) // => { Symbol(a): 1 }
     * load(data, { hashSymbolKeysToString: true }) // => { a: 1 }
     * ```
     */
    hashSymbolKeysToString?: boolean;
    /**
     * Instead of JS object, decode ruby Hash as Map or RubyHash.
     * `hashSymbolKeysToString` is ignored when this option is set.
     * ```js
     * load(data) // => { a: 1 }
     * load(data, { hash: "map" }) // => Map { "a" => 1 }
     * load(data, { hash: "wrap" }) // => RubyHash { entries: [["a", 1]] }
     * ```
     */
    hash?: "map" | "wrap";
    /**
     * Instead of JS regexp, decode ruby Regexp as RubyRegexp.
     * ```js
     * load(data) // => /cat/im
     * load(data, { regexp: "wrap" }) // => RubyRegexp { source: 'cat', options: 5 }
     * ```
     */
    regexp?: "wrap";
    /**
     * If set, put instance variables (often :@key) as string keys in JS objects.
     * If set a string, replace the '@' with the string.
     * Be careful that these ivars won't get dump()ed back.
     * ```js
     * load(data) // => RubyObject { Symbol(@a): 1 }
     * load(data, { ivarToString: true }) // => RubyObject { "@a": 1 }
     * load(data, { ivarToString: "" }) // => RubyObject { "a": 1 }
     * load(data, { ivarToString: "_" }) // => RubyObject { "_a": 1 }
     * ```
     */
    ivarToString?: boolean | string;
    /**
     * If set, use this known classes to decode ruby objects.
     * ```js
     * class A {}
     * load(data) // => RubyObject { class: Symbol(A) }
     * load(data, { known: { A } }) // => A {}
     * ```
     */
    known?: {
        [klass: string]: ClassLike;
    };
}
/**
 * Load one marshal section from buffer.
 *
 * If you need to load multiple times (like RGSS1), use `loadAll`.
 * ```js
 * load(fs.readFileSync('Scripts.rvdata2'))
 * load(await file.arrayBuffer())
 * ```
 */
declare function load(data: string | Uint8Array | ArrayBuffer, options?: LoadOptions): unknown;
declare function loadAll(data: string | Uint8Array | ArrayBuffer, options?: LoadOptions): unknown[];

interface DumpOptions {
    /**
     * If true, convert string keys in JS objects to ruby symbols in Hash.
     * ```js
     * dump({ a: 1 }) // => ruby: { "a" => 1 }
     * dump({ a: 1 }, { hashStringKeysToSymbol: true }) // => ruby: { :a => 1 }
     * ```
     */
    hashStringKeysToSymbol?: boolean;
    /**
     * If set, use this known classes to encode ruby objects.
     * ```js
     * dump(new A()) // => Error "can't dump object [object Object]"
     * dump(new A(), { known: { A } }) // => ruby: #<A>
     * ```
     */
    known?: {
        [klass: string]: ClassLike;
    };
    /**
     * If set, use this string for unknown classes to encode ruby objects.
     * ```js
     * dump(new A()) // => Error "can't dump object [object Object]"
     * dump(new A(), { unknown: () => "A" }) // => ruby: #<A>
     * ```
     */
    unknown?: (obj: unknown) => string | null | undefined;
}
/**
 * Dump a value into marshal buffer.
 * ```js
 * dump(null) // => Uint8Array [ 4, 8, 48 ]
 * ```
 */
declare function dump(value: unknown, options?: DumpOptions): Uint8Array;
declare function dumpAll(value: unknown[], options?: DumpOptions): Uint8Array;

export { type ClassLike, type DumpOptions, type Hash, type LoadOptions, RE_EXTENDED, RE_IGNORECASE, RE_MULTILINE, RubyBaseObject, RubyClass, RubyFloat, RubyHash, RubyInteger, RubyModule, type RubyNumeric, RubyObject, RubyRange, RubyRegexp, RubyStruct, S_DEFAULT, S_EXTENDS, dump, dumpAll, load, loadAll };
