"use strict";
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.findAllReferencesInAst = exports.traversalToReference = void 0;
const syntax_tree_1 = require("./syntax-tree");
function traversalToReference(traversalExpression, localVariables) {
    const meta = traversalExpression.meta;
    // We do not want to include property access through brackets here
    // although it is technically a traversal / reference
    function onlyTakeTraversalPartsUntilFirstBracketPropertyAccess(traversals) {
        let filtered = [];
        let index = 0;
        for (const traversal of traversals) {
            if ("segment" in traversal) {
                if (index === 0) {
                    // We are at the first traversal, check if it is a local variable
                    if (localVariables === null || localVariables === void 0 ? void 0 : localVariables.includes(traversal.segment)) {
                        // We reached a local variable, stop
                        return [];
                    }
                }
                filtered.push(traversal);
            }
            else {
                // We reached a bracket, stop
                return filtered;
            }
        }
        return filtered;
    }
    const filteredParts = onlyTakeTraversalPartsUntilFirstBracketPropertyAccess(meta.traversal);
    if (filteredParts.length === 0) {
        return null;
    }
    const startPosition = filteredParts[0].range.start.byte;
    const endPosition = filteredParts[filteredParts.length - 1].range.end.byte;
    return {
        value: filteredParts.map((part) => part.segment).join("."),
        startPosition,
        endPosition,
    };
}
exports.traversalToReference = traversalToReference;
function findAllReferencesInAst(input, entry, localVariables) {
    if (!entry) {
        return [];
    }
    switch (entry.type) {
        case "scopeTraversal": {
            // For traversals within a for expression,
            // we want to ignore the local variables of the for expression
            const reference = traversalToReference(entry, localVariables);
            if (reference)
                return [reference];
            return [];
        }
        case "for": {
            const meta = entry.meta;
            const additionalLocalVariables = [
                meta.keyVar,
                meta.valVar,
                ...(localVariables || []),
            ];
            return [
                ...findAllReferencesInAst(input, (0, syntax_tree_1.getChildWithValue)(entry, meta.collectionExpression), additionalLocalVariables),
                ...findAllReferencesInAst(input, (0, syntax_tree_1.getChildWithValue)(entry, meta.conditionalExpression), additionalLocalVariables),
            ];
        }
        default:
            return entry.children
                .map((child) => findAllReferencesInAst(input, child, localVariables))
                .flat();
    }
}
exports.findAllReferencesInAst = findAllReferencesInAst;
//# sourceMappingURL=data:application/json;base64,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