"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addParticipants = addParticipants;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const global_1 = require("../../../../config/global");
const logger_1 = require("../../../../logger");
const platform_1 = require("../../../../modules/platform");
const common_1 = require("../../../../util/common");
const sample_1 = require("../../../../util/sample");
const code_owners_1 = require("./code-owners");
async function addCodeOwners(config, assigneesOrReviewers, pr) {
    const codeOwners = await (0, code_owners_1.codeOwnersForPr)(pr);
    const assignees = config.expandCodeOwnersGroups && platform_1.platform.expandGroupMembers
        ? await platform_1.platform.expandGroupMembers(codeOwners)
        : codeOwners;
    return [...new Set(assigneesOrReviewers.concat(assignees))];
}
function filterUnavailableUsers(config, users) {
    return config.filterUnavailableUsers && platform_1.platform.filterUnavailableUsers
        ? platform_1.platform.filterUnavailableUsers(users)
        : Promise.resolve(users);
}
function prepareParticipants(config, usernames) {
    const normalizedUsernames = [...new Set(usernames.map(common_1.noLeadingAtSymbol))];
    return filterUnavailableUsers(config, normalizedUsernames);
}
async function addParticipants(config, pr) {
    let assignees = config.assignees ?? [];
    logger_1.logger.debug(`addParticipants(pr=${pr?.number})`);
    if (config.assigneesFromCodeOwners) {
        assignees = await addCodeOwners(config, assignees, pr);
    }
    if (assignees.length > 0) {
        try {
            assignees = await prepareParticipants(config, assignees);
            if (is_1.default.number(config.assigneesSampleSize)) {
                assignees = (0, sample_1.sampleSize)(assignees, config.assigneesSampleSize);
            }
            if (assignees.length > 0) {
                if (global_1.GlobalConfig.get('dryRun')) {
                    logger_1.logger.info(`DRY-RUN: Would add assignees to PR #${pr.number}`);
                }
                else {
                    await platform_1.platform.addAssignees(pr.number, assignees);
                    logger_1.logger.debug({ assignees }, 'Added assignees');
                }
            }
        }
        catch (err) {
            logger_1.logger.debug({ assignees: config.assignees, err }, 'Failed to add assignees');
        }
    }
    let reviewers = config.reviewers ?? [];
    if (config.reviewersFromCodeOwners) {
        reviewers = await addCodeOwners(config, reviewers, pr);
        logger_1.logger.debug(`Reviewers from code owners: ${reviewers.map((reviewer) => `"${reviewer}"`).join(', ')}`);
    }
    if (is_1.default.array(config.additionalReviewers) &&
        config.additionalReviewers.length > 0) {
        logger_1.logger.debug(`Additional reviewers: ${config.additionalReviewers.map((reviewer) => `"${reviewer}"`).join(', ')}`);
        reviewers = reviewers.concat(config.additionalReviewers);
    }
    if (reviewers.length > 0) {
        try {
            reviewers = await prepareParticipants(config, reviewers);
            if (is_1.default.number(config.reviewersSampleSize)) {
                logger_1.logger.debug(`Sampling reviewersSampleSize=${config.reviewersSampleSize} reviewers`);
                reviewers = (0, sample_1.sampleSize)(reviewers, config.reviewersSampleSize);
            }
            if (reviewers.length > 0) {
                if (global_1.GlobalConfig.get('dryRun')) {
                    logger_1.logger.info(`DRY-RUN: Would add reviewers to PR #${pr.number}`);
                }
                else {
                    await platform_1.platform.addReviewers(pr.number, reviewers);
                    logger_1.logger.debug({ reviewers }, 'Added reviewers');
                }
            }
        }
        catch (err) {
            logger_1.logger.debug({ reviewers: config.reviewers, err }, 'Failed to add reviewers');
        }
    }
}
//# sourceMappingURL=participants.js.map