"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveBranchStatus = resolveBranchStatus;
exports.setStability = setStability;
exports.setConfidence = setConfidence;
const logger_1 = require("../../../../logger");
const platform_1 = require("../../../../modules/platform");
const merge_confidence_1 = require("../../../../util/merge-confidence");
const string_1 = require("../../../../util/string");
const url_1 = require("../../../../util/url");
async function resolveBranchStatus(branchName, internalChecksAsSuccess, ignoreTests = false) {
    logger_1.logger.debug(`resolveBranchStatus(branchName=${branchName}, ignoreTests=${ignoreTests})`);
    if (ignoreTests) {
        logger_1.logger.debug('Ignore tests. Return green');
        return 'green';
    }
    const status = await platform_1.platform.getBranchStatus(branchName, internalChecksAsSuccess);
    logger_1.logger.debug(`Branch status ${status}`);
    return status;
}
async function setStatusCheck(branchName, context, description, state, url) {
    const existingState = await platform_1.platform.getBranchStatusCheck(branchName, context);
    if (existingState === state) {
        logger_1.logger.debug(`Status check ${context} is already up-to-date`);
    }
    else {
        logger_1.logger.debug(`Updating ${context} status check state to ${state}`);
        await platform_1.platform.setBranchStatus({
            branchName,
            context,
            description,
            state,
            url,
        });
    }
}
async function setStability(config) {
    if (!config.stabilityStatus) {
        return;
    }
    const context = config.statusCheckNames?.minimumReleaseAge;
    if (!context) {
        logger_1.logger.debug('Status check is null or an empty string, skipping status check addition.');
        return;
    }
    const description = config.stabilityStatus === 'green'
        ? 'Updates have met minimum release age requirement'
        : 'Updates have not met minimum release age requirement';
    const docsLink = (0, url_1.joinUrlParts)((0, string_1.coerceString)(config.productLinks?.documentation), 'configuration-options/#minimumreleaseage');
    await setStatusCheck(config.branchName, context, description, config.stabilityStatus, docsLink);
}
async function setConfidence(config) {
    if (!config.branchName ||
        !config.confidenceStatus ||
        (config.minimumConfidence &&
            !(0, merge_confidence_1.isActiveConfidenceLevel)(config.minimumConfidence))) {
        return;
    }
    const context = config.statusCheckNames?.mergeConfidence;
    if (!context) {
        logger_1.logger.debug('Status check is null or an empty string, skipping status check addition.');
        return;
    }
    const description = config.confidenceStatus === 'green'
        ? 'Updates have met Merge Confidence requirement'
        : 'Updates have not met Merge Confidence requirement';
    const docsLink = (0, url_1.joinUrlParts)((0, string_1.coerceString)(config.productLinks?.documentation), 'merge-confidence');
    await setStatusCheck(config.branchName, context, description, config.confidenceStatus, docsLink);
}
//# sourceMappingURL=status-checks.js.map