"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateReconfigureBranch = validateReconfigureBranch;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const massage_1 = require("../../../config/massage");
const validation_1 = require("../../../config/validation");
const logger_1 = require("../../../logger");
const platform_1 = require("../../../modules/platform");
const comment_1 = require("../../../modules/platform/comment");
const git_1 = require("../../../util/git");
const regex_1 = require("../../../util/regex");
const reconfigure_cache_1 = require("./reconfigure-cache");
const utils_1 = require("./utils");
async function validateReconfigureBranch(config, reconfigureConfig, configFileName, reconfigurePr) {
    logger_1.logger.debug('validateReconfigureBranch()');
    const context = config.statusCheckNames?.configValidation;
    const branchName = (0, utils_1.getReconfigureBranchName)(config.branchPrefix);
    const branchSha = (0, git_1.getBranchCommit)(branchName);
    if (context) {
        const validationStatus = await platform_1.platform.getBranchStatusCheck(branchName, context);
        // if old status check is present skip validation
        if (is_1.default.nonEmptyString(validationStatus)) {
            logger_1.logger.debug('Skipping validation check because status check already exists.');
            return validationStatus === 'green';
        }
    }
    else {
        logger_1.logger.debug('Status check is null or an empty string, skipping status check addition.');
    }
    // perform validation and provide a passing or failing check based on result
    const massagedConfig = (0, massage_1.massageConfig)(reconfigureConfig);
    const validationResult = await (0, validation_1.validateConfig)('repo', massagedConfig);
    // failing check
    if (validationResult.errors.length > 0) {
        logger_1.logger.debug({ errors: validationResult.errors.map((err) => err.message).join(', ') }, 'Validation Errors');
        // add comment to reconfigure PR if it exists
        if (reconfigurePr) {
            let body = `There is an error with this repository's Renovate configuration that needs to be fixed.\n\n`;
            body += `Location: \`${configFileName}\`\n`;
            body += `Message: \`${validationResult.errors
                .map((e) => e.message)
                .join(', ')
                .replace((0, regex_1.regEx)(/`/g), "'")}\`\n`;
            await (0, comment_1.ensureComment)({
                number: reconfigurePr.number,
                topic: 'Action Required: Fix Renovate Configuration',
                content: body,
            });
        }
        await (0, utils_1.setBranchStatus)(branchName, 'Validation Failed', 'red', context);
        (0, reconfigure_cache_1.setReconfigureBranchCache)(branchSha, false);
        return false;
    }
    // passing check
    await (0, utils_1.setBranchStatus)(branchName, 'Validation Successful', 'green', context);
    return true;
}
//# sourceMappingURL=validate.js.map