"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.calculateAbandonment = calculateAbandonment;
const luxon_1 = require("luxon");
const logger_1 = require("../../../../logger");
const pretty_time_1 = require("../../../../util/pretty-time");
const stats_1 = require("../../../../util/stats");
function calculateAbandonment(releaseResult, config) {
    const { lookupName } = releaseResult;
    const { abandonmentThreshold } = config;
    if (!abandonmentThreshold) {
        logger_1.logger.trace({ lookupName }, 'No abandonmentThreshold defined, skipping abandonment check');
        return releaseResult;
    }
    const abandonmentThresholdMs = (0, pretty_time_1.toMs)(abandonmentThreshold);
    if (!abandonmentThresholdMs) {
        logger_1.logger.trace({ lookupName, abandonmentThreshold }, 'Could not parse abandonmentThreshold to milliseconds, skipping abandonment check');
        return releaseResult;
    }
    const { mostRecentTimestamp } = releaseResult;
    if (!mostRecentTimestamp) {
        logger_1.logger.trace({ lookupName }, 'No mostRecentTimestamp value found, skipping abandonment check');
        return releaseResult;
    }
    const mostRecentTimestampDate = luxon_1.DateTime.fromISO(mostRecentTimestamp);
    const abandonmentDate = mostRecentTimestampDate.plus({
        milliseconds: abandonmentThresholdMs,
    });
    const now = luxon_1.DateTime.local();
    const isAbandoned = abandonmentDate < now;
    releaseResult.isAbandoned = isAbandoned;
    if (isAbandoned) {
        logger_1.logger.debug(`Package abandonment detected: ${config.packageName} (${config.datasource}) - most recent release: ${mostRecentTimestamp}`);
    }
    logger_1.logger.trace({
        lookupName,
        mostRecentTimestamp,
        abandonmentThreshold,
        abandonmentThresholdMs,
        abandonmentDate: abandonmentDate.toISO(),
        now: now.toISO(),
        isAbandoned,
    }, 'Calculated abandonment status');
    if (isAbandoned) {
        const { datasource, packageName } = config;
        stats_1.AbandonedPackageStats.write(datasource, packageName, mostRecentTimestamp);
    }
    return releaseResult;
}
//# sourceMappingURL=abandonment.js.map