"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPrHourlyCount = getPrHourlyCount;
exports.getConcurrentPrsCount = getConcurrentPrsCount;
exports.getConcurrentBranchesCount = getConcurrentBranchesCount;
const luxon_1 = require("luxon");
const logger_1 = require("../../../logger");
const platform_1 = require("../../../modules/platform");
const scm_1 = require("../../../modules/platform/scm");
const external_host_error_1 = require("../../../types/errors/external-host-error");
async function getPrHourlyCount(config) {
    try {
        const prList = await platform_1.platform.getPrList();
        const currentHourStart = luxon_1.DateTime.local().setZone('utc').startOf('hour');
        logger_1.logger.debug(`Calculating PRs created so far in this hour currentHourStart=${String(currentHourStart)}`);
        const soFarThisHour = prList.filter((pr) => pr.sourceBranch !== config.onboardingBranch &&
            pr.sourceBranch.startsWith(config.branchPrefix) &&
            luxon_1.DateTime.fromISO(pr.createdAt) > currentHourStart);
        logger_1.logger.debug(`${soFarThisHour.length} PRs have been created so far in this hour.`);
        return soFarThisHour.length;
    }
    catch (err) {
        // istanbul ignore if
        if (err instanceof external_host_error_1.ExternalHostError) {
            throw err;
        }
        logger_1.logger.error({ err }, 'Error checking PRs created per hour');
        return 0;
    }
}
async function getConcurrentPrsCount(config, branches) {
    let openPrCount = 0;
    for (const { branchName } of branches) {
        try {
            const pr = await platform_1.platform.getBranchPr(branchName, config.baseBranch);
            if (pr &&
                pr.sourceBranch !== config.onboardingBranch &&
                pr.state === 'open') {
                openPrCount++;
            }
        }
        catch (err) {
            // istanbul ignore if
            if (err instanceof external_host_error_1.ExternalHostError) {
                throw err;
            }
            else {
                // no-op
            }
        }
    }
    logger_1.logger.debug(`${openPrCount} PRs are currently open`);
    return openPrCount;
}
async function getConcurrentBranchesCount(branches) {
    let existingBranchCount = 0;
    for (const branch of branches) {
        if (await scm_1.scm.branchExists(branch.branchName)) {
            existingBranchCount++;
        }
    }
    logger_1.logger.debug(`${existingBranchCount} already existing branches found.`);
    return existingBranchCount;
}
//# sourceMappingURL=limits.js.map