"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractDependencies = extractDependencies;
exports.updateRepo = updateRepo;
// TODO #22198
const config_1 = require("../../../config");
const global_1 = require("../../../config/global");
const presets_1 = require("../../../config/presets");
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const platform_1 = require("../../../modules/platform");
const scm_1 = require("../../../modules/platform/scm");
const repository_1 = require("../../../util/cache/repository");
const clone_1 = require("../../../util/clone");
const git_1 = require("../../../util/git");
const split_1 = require("../../../util/split");
const string_match_1 = require("../../../util/string-match");
const dependency_dashboard_1 = require("../dependency-dashboard");
const extract_update_1 = require("./extract-update");
async function getBaseBranchConfig(baseBranch, config) {
    logger_1.logger.debug(`baseBranch: ${baseBranch}`);
    let baseBranchConfig = (0, clone_1.clone)(config);
    if (config.useBaseBranchConfig === 'merge' &&
        baseBranch !== config.defaultBranch) {
        logger_1.logger.debug({ baseBranch }, `Merging config from base branch because useBaseBranchConfig=merge`);
        // Retrieve config file name autodetected for this repo
        const cache = (0, repository_1.getCache)();
        // TODO: types (#22198)
        const configFileName = cache.configFileName;
        try {
            baseBranchConfig = await platform_1.platform.getJsonFile(configFileName, config.repository, baseBranch);
            logger_1.logger.debug({ config: baseBranchConfig }, 'Base branch config raw');
        }
        catch {
            logger_1.logger.error({ configFileName, baseBranch }, `Error fetching config file from base branch - possible config name mismatch between branches?`);
            const error = new Error(error_messages_1.CONFIG_VALIDATION);
            error.validationSource = 'config';
            error.validationError = 'Error fetching config file';
            error.validationMessage = `Error fetching config file \`${configFileName}\` from branch \`${baseBranch}\``;
            throw error;
        }
        baseBranchConfig = await (0, presets_1.resolveConfigPresets)(baseBranchConfig, config);
        baseBranchConfig = (0, config_1.mergeChildConfig)(config, baseBranchConfig);
        // istanbul ignore if
        if (config.printConfig) {
            logger_1.logger.info({ config: baseBranchConfig }, 'Base branch config after merge');
        }
        // baseBranches value should be based off the default branch
        baseBranchConfig.baseBranches = config.baseBranches;
    }
    if (config.baseBranches.length > 1) {
        baseBranchConfig.branchPrefix += `${baseBranch}-`;
        baseBranchConfig.hasBaseBranches = true;
    }
    baseBranchConfig = (0, config_1.mergeChildConfig)(baseBranchConfig, { baseBranch });
    return baseBranchConfig;
}
function unfoldBaseBranches(defaultBranch, baseBranches) {
    const unfoldedList = [];
    const allBranches = (0, git_1.getBranchList)();
    for (const baseBranch of baseBranches) {
        const isAllowedPred = (0, string_match_1.getRegexPredicate)(baseBranch);
        if (isAllowedPred) {
            const matchingBranches = allBranches.filter(isAllowedPred);
            logger_1.logger.debug(`baseBranches regex "${baseBranch}" matches [${matchingBranches.join()}]`);
            unfoldedList.push(...matchingBranches);
        }
        else if (baseBranch === '$default') {
            logger_1.logger.debug(`baseBranches "$default" matches "${defaultBranch}"`);
            unfoldedList.push(defaultBranch);
        }
        else {
            unfoldedList.push(baseBranch);
        }
    }
    return [...new Set(unfoldedList)];
}
async function extractDependencies(config, overwriteCache = true) {
    await (0, dependency_dashboard_1.readDashboardBody)(config);
    let res = {
        branches: [],
        branchList: [],
        packageFiles: {},
    };
    if (global_1.GlobalConfig.get('platform') !== 'local' && config.baseBranches?.length) {
        config.baseBranches = unfoldBaseBranches(config.defaultBranch, config.baseBranches);
        logger_1.logger.debug({ baseBranches: config.baseBranches }, 'baseBranches');
        const extracted = {};
        for (const baseBranch of config.baseBranches) {
            (0, logger_1.addMeta)({ baseBranch });
            if (scm_1.scm.syncForkWithUpstream) {
                await scm_1.scm.syncForkWithUpstream(baseBranch);
            }
            if (await scm_1.scm.branchExists(baseBranch)) {
                const baseBranchConfig = await getBaseBranchConfig(baseBranch, config);
                extracted[baseBranch] = await (0, extract_update_1.extract)(baseBranchConfig, overwriteCache);
            }
            else {
                logger_1.logger.warn({ baseBranch }, 'Base branch does not exist - skipping');
            }
        }
        (0, split_1.addSplit)('extract');
        for (const baseBranch of config.baseBranches) {
            if (await scm_1.scm.branchExists(baseBranch)) {
                (0, logger_1.addMeta)({ baseBranch });
                const baseBranchConfig = await getBaseBranchConfig(baseBranch, config);
                const packageFiles = extracted[baseBranch];
                const baseBranchRes = await (0, extract_update_1.lookup)(baseBranchConfig, packageFiles);
                res.branches = res.branches.concat(baseBranchRes?.branches);
                res.branchList = res.branchList.concat(baseBranchRes?.branchList);
                if (!res.packageFiles || !Object.keys(res.packageFiles).length) {
                    // Use the first branch
                    res.packageFiles = baseBranchRes?.packageFiles;
                }
            }
        }
        (0, logger_1.removeMeta)(['baseBranch']);
    }
    else {
        logger_1.logger.debug('No baseBranches');
        const packageFiles = await (0, extract_update_1.extract)(config, overwriteCache);
        (0, split_1.addSplit)('extract');
        if (global_1.GlobalConfig.get('dryRun') === 'extract') {
            res.packageFiles = packageFiles;
            logger_1.logger.info({ packageFiles }, 'Extracted dependencies');
            return res;
        }
        res = await (0, extract_update_1.lookup)(config, packageFiles);
    }
    (0, split_1.addSplit)('lookup');
    return res;
}
function updateRepo(config, branches) {
    logger_1.logger.debug('processRepo()');
    return (0, extract_update_1.update)(config, branches);
}
//# sourceMappingURL=index.js.map