"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.fetchUpdates = fetchUpdates;
const tslib_1 = require("tslib");
// TODO #22198
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const config_1 = require("../../../config");
const logger_1 = require("../../../logger");
const datasource_1 = require("../../../modules/datasource");
const common_1 = require("../../../modules/datasource/common");
const external_host_error_1 = require("../../../types/errors/external-host-error");
const clone_1 = require("../../../util/clone");
const package_rules_1 = require("../../../util/package-rules");
const p = tslib_1.__importStar(require("../../../util/promises"));
const result_1 = require("../../../util/result");
const stats_1 = require("../../../util/stats");
const package_files_1 = require("../package-files");
const lookup_1 = require("./lookup");
async function lookup(packageFileConfig, indep) {
    const dep = (0, clone_1.clone)(indep);
    dep.updates = [];
    if (is_1.default.string(dep.depName)) {
        dep.depName = dep.depName.trim();
    }
    dep.packageName ??= dep.depName;
    if (dep.skipReason) {
        return result_1.Result.ok(dep);
    }
    if (!is_1.default.nonEmptyString(dep.packageName)) {
        dep.skipReason = 'invalid-name';
        return result_1.Result.ok(dep);
    }
    if (dep.isInternal && !packageFileConfig.updateInternalDeps) {
        dep.skipReason = 'internal-package';
        return result_1.Result.ok(dep);
    }
    const { depName } = dep;
    // TODO: fix types
    let depConfig = (0, config_1.mergeChildConfig)(packageFileConfig, dep);
    const datasourceDefaultConfig = await (0, datasource_1.getDefaultConfig)(depConfig.datasource);
    depConfig = (0, config_1.mergeChildConfig)(depConfig, datasourceDefaultConfig);
    depConfig.versioning ??= (0, common_1.getDefaultVersioning)(depConfig.datasource);
    depConfig = await (0, package_rules_1.applyPackageRules)(depConfig, 'pre-lookup');
    depConfig.packageName ??= depConfig.depName;
    if (depConfig.ignoreDeps.includes(depName)) {
        // TODO: fix types (#22198)
        logger_1.logger.debug(`Dependency: ${depName}, is ignored`);
        dep.skipReason = 'ignored';
        return result_1.Result.ok(dep);
    }
    if (depConfig.enabled === false) {
        logger_1.logger.debug(`Dependency: ${depName}, is disabled`);
        dep.skipReason = 'disabled';
        return result_1.Result.ok(dep);
    }
    if (!depConfig.datasource) {
        return result_1.Result.ok(dep);
    }
    return stats_1.LookupStats.wrap(depConfig.datasource, async () => {
        return await result_1.Result.wrap((0, lookup_1.lookupUpdates)(depConfig))
            .onValue((dep) => {
            logger_1.logger.trace({ dep }, 'Dependency lookup success');
        })
            .onError((err) => {
            logger_1.logger.trace({ err, depName }, 'Dependency lookup error');
        })
            .catch((err) => {
            if (packageFileConfig.repoIsOnboarded === true ||
                !(err instanceof external_host_error_1.ExternalHostError)) {
                return result_1.Result.err(err);
            }
            const cause = err.err;
            return result_1.Result.ok({
                updates: [],
                warnings: [
                    {
                        topic: 'Lookup Error',
                        message: `${depName}: ${cause.message}`,
                    },
                ],
            });
        })
            .transform((upd) => Object.assign(dep, upd));
    });
}
async function fetchManagerPackagerFileUpdates(config, managerConfig, pFile) {
    const { packageFile } = pFile;
    const packageFileConfig = (0, config_1.mergeChildConfig)(managerConfig, pFile);
    if (pFile.extractedConstraints) {
        packageFileConfig.constraints = {
            ...pFile.extractedConstraints,
            ...config.constraints,
        };
    }
    const { manager } = packageFileConfig;
    const queue = pFile.deps.map((dep) => async () => {
        const updates = await lookup(packageFileConfig, dep);
        return updates.unwrapOrThrow();
    });
    logger_1.logger.trace({ manager, packageFile, queueLength: queue.length }, 'fetchManagerPackagerFileUpdates starting with concurrency');
    pFile.deps = await p.all(queue);
    logger_1.logger.trace({ packageFile }, 'fetchManagerPackagerFileUpdates finished');
}
async function fetchManagerUpdates(config, packageFiles, manager) {
    const managerConfig = (0, config_1.getManagerConfig)(config, manager);
    const queue = packageFiles[manager].map((pFile) => () => fetchManagerPackagerFileUpdates(config, managerConfig, pFile));
    logger_1.logger.trace({ manager, queueLength: queue.length }, 'fetchManagerUpdates starting');
    await p.all(queue);
    logger_1.logger.trace({ manager }, 'fetchManagerUpdates finished');
}
async function fetchUpdates(config, packageFiles) {
    const managers = Object.keys(packageFiles);
    const allManagerJobs = managers.map((manager) => fetchManagerUpdates(config, packageFiles, manager));
    await Promise.all(allManagerJobs);
    package_files_1.PackageFiles.add(config.baseBranch, { ...packageFiles });
    logger_1.logger.debug({ baseBranch: config.baseBranch }, 'Package releases lookups complete');
}
//# sourceMappingURL=fetch.js.map