"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ensureOnboardingPr = ensureOnboardingPr;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const global_1 = require("../../../../config/global");
const logger_1 = require("../../../../logger");
const platform_1 = require("../../../../modules/platform");
const comment_1 = require("../../../../modules/platform/comment");
const pr_body_1 = require("../../../../modules/platform/pr-body");
const scm_1 = require("../../../../modules/platform/scm");
const emoji_1 = require("../../../../util/emoji");
const git_1 = require("../../../../util/git");
const hash_1 = require("../../../../util/hash");
const template = tslib_1.__importStar(require("../../../../util/template"));
const errors_warnings_1 = require("../../errors-warnings");
const pr_1 = require("../../update/pr");
const labels_1 = require("../../update/pr/labels");
const participants_1 = require("../../update/pr/participants");
const onboarding_branch_cache_1 = require("../branch/onboarding-branch-cache");
const common_1 = require("../common");
const base_branch_1 = require("./base-branch");
const config_description_1 = require("./config-description");
const pr_list_1 = require("./pr-list");
async function ensureOnboardingPr(config, packageFiles, branches) {
    if (config.repoIsOnboarded === true ||
        common_1.OnboardingState.onboardingCacheValid ||
        (config.onboardingRebaseCheckbox && !common_1.OnboardingState.prUpdateRequested)) {
        return;
    }
    logger_1.logger.debug('ensureOnboardingPr()');
    logger_1.logger.trace({ config });
    // TODO #22198
    const existingPr = await platform_1.platform.getBranchPr(config.onboardingBranch, config.defaultBranch);
    if (existingPr) {
        // skip pr-update if branch is conflicted
        if (await (0, onboarding_branch_cache_1.isOnboardingBranchConflicted)(config.defaultBranch, config.onboardingBranch)) {
            if (global_1.GlobalConfig.get('dryRun')) {
                logger_1.logger.info('DRY-RUN: Would comment that Onboarding PR is conflicted and needs manual resolving');
                return;
            }
            await (0, comment_1.ensureComment)({
                number: existingPr.number,
                topic: 'Branch Conflicted',
                content: (0, emoji_1.emojify)(`:warning: This PR has a merge conflict which Renovate is unable to automatically resolve, so updates to this PR description are now paused. Please resolve the merge conflict manually.\n\n`),
            });
            return;
        }
    }
    const onboardingConfigHashComment = await getOnboardingConfigHashComment(config);
    const rebaseCheckBox = getRebaseCheckbox(config.onboardingRebaseCheckbox);
    logger_1.logger.debug('Filling in onboarding PR template');
    let prTemplate = `Welcome to [Renovate](${config.productLinks.homepage})! This is an onboarding PR to help you understand and configure settings before regular Pull Requests begin.\n\n`;
    prTemplate +=
        config.requireConfig === 'required'
            ? (0, emoji_1.emojify)(`:vertical_traffic_light: To activate Renovate, merge this Pull Request. To disable Renovate, simply close this Pull Request unmerged.\n\n`)
            : (0, emoji_1.emojify)(`:vertical_traffic_light: Renovate will begin keeping your dependencies up-to-date only once you merge or close this Pull Request.\n\n`);
    // TODO #22198
    prTemplate += (0, emoji_1.emojify)(`

---
{{PACKAGE FILES}}
{{CONFIG}}
{{BASEBRANCH}}
{{PRLIST}}
{{WARNINGS}}
{{ERRORS}}

---

:question: Got questions? Check out Renovate's [Docs](${config.productLinks.documentation}), particularly the Getting Started section.
If you need any further assistance then you can also [request help here](${config.productLinks.help}).
`);
    prTemplate += rebaseCheckBox;
    let prBody = prTemplate;
    if (packageFiles && Object.entries(packageFiles).length) {
        let files = [];
        for (const [manager, managerFiles] of Object.entries(packageFiles)) {
            files = files.concat(managerFiles.map((file) => ` * \`${file.packageFile}\` (${manager})`));
        }
        prBody =
            prBody.replace('{{PACKAGE FILES}}', '### Detected Package Files\n\n' + files.join('\n')) + '\n';
    }
    else {
        prBody = prBody.replace('{{PACKAGE FILES}}\n', '');
    }
    let configDesc = '';
    if (global_1.GlobalConfig.get('dryRun')) {
        // TODO: types (#22198)
        logger_1.logger.info(`DRY-RUN: Would check branch ${config.onboardingBranch}`);
    }
    else {
        configDesc = (0, config_description_1.getConfigDesc)(config, packageFiles);
    }
    prBody = prBody.replace('{{CONFIG}}\n', configDesc);
    prBody = prBody.replace('{{WARNINGS}}\n', (0, errors_warnings_1.getWarnings)(config) + (0, errors_warnings_1.getDepWarningsOnboardingPR)(packageFiles, config));
    prBody = prBody.replace('{{ERRORS}}\n', (0, errors_warnings_1.getErrors)(config));
    prBody = prBody.replace('{{BASEBRANCH}}\n', (0, base_branch_1.getBaseBranchDesc)(config));
    prBody = prBody.replace('{{PRLIST}}\n', (0, pr_list_1.getExpectedPrList)(config, branches));
    if (is_1.default.string(config.prHeader)) {
        prBody = `${template.compile(config.prHeader, config)}\n\n${prBody}`;
    }
    if (is_1.default.string(config.prFooter)) {
        prBody = `${prBody}\n---\n\n${template.compile(config.prFooter, config)}\n`;
    }
    prBody += onboardingConfigHashComment;
    logger_1.logger.trace('prBody:\n' + prBody);
    prBody = platform_1.platform.massageMarkdown(prBody, config.rebaseLabel);
    if (existingPr) {
        logger_1.logger.debug('Found open onboarding PR');
        // Check if existing PR needs updating
        const prBodyHash = (0, pr_body_1.hashBody)(prBody);
        if (existingPr.bodyStruct?.hash === prBodyHash) {
            logger_1.logger.debug(`Pull Request #${existingPr.number} does not need updating`);
            return;
        }
        // PR must need updating
        if (global_1.GlobalConfig.get('dryRun')) {
            logger_1.logger.info('DRY-RUN: Would update onboarding PR');
        }
        else {
            await platform_1.platform.updatePr({
                number: existingPr.number,
                prTitle: existingPr.title,
                prBody,
            });
            logger_1.logger.info({ pr: existingPr.number }, 'Onboarding PR updated');
        }
        return;
    }
    logger_1.logger.debug('Creating onboarding PR');
    const labels = (0, labels_1.prepareLabels)(config);
    try {
        if (global_1.GlobalConfig.get('dryRun')) {
            logger_1.logger.info('DRY-RUN: Would create onboarding PR');
        }
        else {
            // TODO #22198
            const prTitle = config.semanticCommits === 'enabled'
                ? (0, common_1.getSemanticCommitPrTitle)(config)
                : config.onboardingPrTitle;
            const pr = await platform_1.platform.createPr({
                sourceBranch: config.onboardingBranch,
                targetBranch: config.defaultBranch,
                prTitle,
                prBody,
                labels,
                platformPrOptions: (0, pr_1.getPlatformPrOptions)({
                    ...config,
                    automerge: false,
                }),
            });
            logger_1.logger.info({ pr: `Pull Request #${pr.number}` }, 'Onboarding PR created');
            await (0, participants_1.addParticipants)(config, pr);
        }
    }
    catch (err) {
        if (err.response?.statusCode === 422 &&
            err.response?.body?.errors?.[0]?.message?.startsWith('A pull request already exists')) {
            logger_1.logger.warn('Onboarding PR already exists but cannot find it. It was probably created by a different user.');
            await scm_1.scm.deleteBranch(config.onboardingBranch);
            return;
        }
        throw err;
    }
}
function getRebaseCheckbox(onboardingRebaseCheckbox) {
    let rebaseCheckBox = '';
    if (onboardingRebaseCheckbox) {
        // Create markdown checkbox
        rebaseCheckBox = `\n\n---\n\n - [ ] <!-- rebase-check -->If you want to rebase/retry this PR, click this checkbox.\n`;
    }
    return rebaseCheckBox;
}
async function getOnboardingConfigHashComment(config) {
    const configFile = (0, common_1.getDefaultConfigFileName)(config);
    const existingContents = (await (0, git_1.getFile)(configFile, config.onboardingBranch)) ?? '';
    const hash = (0, hash_1.toSha256)(existingContents);
    return `\n<!--renovate-config-hash:${hash}-->\n`;
}
//# sourceMappingURL=index.js.map