"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkOnboardingBranch = checkOnboardingBranch;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const config_1 = require("../../../../config");
const global_1 = require("../../../../config/global");
const error_messages_1 = require("../../../../constants/error-messages");
const logger_1 = require("../../../../logger");
const platform_1 = require("../../../../modules/platform");
const scm_1 = require("../../../../modules/platform/scm");
const repository_1 = require("../../../../util/cache/repository");
const git_1 = require("../../../../util/git");
const configured_1 = require("../../configured");
const extract_1 = require("../../extract");
const merge_1 = require("../../init/merge");
const common_1 = require("../common");
const check_1 = require("./check");
const config_2 = require("./config");
const create_1 = require("./create");
const onboarding_branch_cache_1 = require("./onboarding-branch-cache");
const rebase_1 = require("./rebase");
async function checkOnboardingBranch(config) {
    logger_1.logger.debug('checkOnboarding()');
    logger_1.logger.trace({ config });
    let onboardingBranch = config.onboardingBranch;
    const defaultBranch = config.defaultBranch;
    let isConflicted = false;
    let isModified = false;
    const repoIsOnboarded = await (0, check_1.isOnboarded)(config);
    if (repoIsOnboarded) {
        logger_1.logger.debug('Repo is onboarded');
        // delete onboarding cache
        (0, onboarding_branch_cache_1.deleteOnboardingCache)();
        return { ...config, repoIsOnboarded };
    }
    (0, configured_1.checkIfConfigured)(config);
    logger_1.logger.debug('Repo is not onboarded');
    // global gitAuthor will need to be used
    (0, git_1.setGitAuthor)(config.gitAuthor);
    const onboardingPr = await (0, check_1.getOnboardingPr)(config);
    // TODO #22198
    const branchList = [onboardingBranch];
    if (onboardingPr) {
        logger_1.logger.debug('Onboarding PR already exists');
        isModified = await (0, onboarding_branch_cache_1.isOnboardingBranchModified)(config.onboardingBranch, defaultBranch);
        // if onboarding branch is not modified, check if onboarding config has been changed and rebase if true
        if (!isModified) {
            const commit = await (0, rebase_1.rebaseOnboardingBranch)(config, onboardingPr.bodyStruct?.rawConfigHash);
            if (commit) {
                logger_1.logger.info({ branch: config.onboardingBranch, commit, onboarding: true }, 'Branch updated');
            }
            // istanbul ignore if
            if (platform_1.platform.refreshPr) {
                await platform_1.platform.refreshPr(onboardingPr.number);
            }
        }
        if (config.onboardingRebaseCheckbox) {
            handleOnboardingManualRebase(onboardingPr);
        }
        if (isConfigHashPresent(onboardingPr) && // needed so that existing onboarding PRs are updated with config hash comment
            isOnboardingCacheValid(defaultBranch, config.onboardingBranch) &&
            !(config.onboardingRebaseCheckbox && common_1.OnboardingState.prUpdateRequested)) {
            logger_1.logger.debug('Skip processing since the onboarding branch is up to date and default branch has not changed');
            common_1.OnboardingState.onboardingCacheValid = true;
            return { ...config, repoIsOnboarded, onboardingBranch, branchList };
        }
        common_1.OnboardingState.onboardingCacheValid = false;
        if (isModified) {
            if ((0, onboarding_branch_cache_1.hasOnboardingBranchChanged)(config.onboardingBranch)) {
                invalidateExtractCache(config.baseBranch);
            }
            isConflicted = await (0, onboarding_branch_cache_1.isOnboardingBranchConflicted)(config.baseBranch, config.onboardingBranch);
        }
    }
    else {
        logger_1.logger.debug('Onboarding PR does not exist');
        const onboardingConfig = await (0, config_2.getOnboardingConfig)(config);
        let mergedConfig = (0, config_1.mergeChildConfig)(config, onboardingConfig);
        mergedConfig = await (0, merge_1.mergeRenovateConfig)(mergedConfig);
        onboardingBranch = mergedConfig.onboardingBranch;
        if (Object.entries((await (0, extract_1.extractAllDependencies)(mergedConfig)).packageFiles)
            .length === 0) {
            if (config.onboardingNoDeps !== 'enabled') {
                throw new Error(error_messages_1.REPOSITORY_NO_PACKAGE_FILES);
            }
        }
        logger_1.logger.debug('Need to create onboarding PR');
        if (config.onboardingRebaseCheckbox) {
            common_1.OnboardingState.prUpdateRequested = true;
        }
        const commit = await (0, create_1.createOnboardingBranch)(mergedConfig);
        // istanbul ignore if
        if (commit) {
            logger_1.logger.info({ branch: onboardingBranch, commit, onboarding: true }, 'Branch created');
        }
    }
    if (!global_1.GlobalConfig.get('dryRun')) {
        // TODO #22198
        if (!isConflicted) {
            logger_1.logger.debug('Merge onboarding branch in default branch');
            await scm_1.scm.mergeToLocal(onboardingBranch);
        }
    }
    (0, onboarding_branch_cache_1.setOnboardingCache)((0, git_1.getBranchCommit)(config.defaultBranch), (0, git_1.getBranchCommit)(onboardingBranch), isConflicted, isModified);
    return { ...config, repoIsOnboarded, onboardingBranch, branchList };
}
function handleOnboardingManualRebase(onboardingPr) {
    const pl = global_1.GlobalConfig.get('platform');
    const { rebaseRequested } = onboardingPr.bodyStruct ?? {};
    if (!['github', 'gitlab', 'gitea'].includes(pl)) {
        logger_1.logger.trace(`Platform '${pl}' does not support extended markdown`);
        common_1.OnboardingState.prUpdateRequested = true;
    }
    else if (is_1.default.nullOrUndefined(rebaseRequested)) {
        logger_1.logger.debug('No rebase checkbox was found in the onboarding PR');
        common_1.OnboardingState.prUpdateRequested = true;
    }
    else if (rebaseRequested) {
        logger_1.logger.debug('Manual onboarding PR update requested');
        common_1.OnboardingState.prUpdateRequested = true;
    }
}
function invalidateExtractCache(baseBranch) {
    const cache = (0, repository_1.getCache)();
    cache.scan ??= {};
    if (cache.scan?.[baseBranch]) {
        delete cache.scan[baseBranch];
    }
}
function isOnboardingCacheValid(defaultBranch, onboardingBranch) {
    const cache = (0, repository_1.getCache)();
    const onboardingBranchCache = cache?.onboardingBranchCache;
    return !!(onboardingBranchCache &&
        onboardingBranchCache.defaultBranchSha === (0, git_1.getBranchCommit)(defaultBranch) &&
        onboardingBranchCache.onboardingBranchSha ===
            (0, git_1.getBranchCommit)(onboardingBranch) &&
        onboardingBranchCache.configFileName &&
        onboardingBranchCache.configFileParsed);
}
function isConfigHashPresent(pr) {
    const platform = global_1.GlobalConfig.get('platform');
    // if platform does not support html comments return true
    if (!['github', 'gitlab', 'gitea'].includes(platform)) {
        return true;
    }
    return !!pr.bodyStruct?.rawConfigHash;
}
//# sourceMappingURL=index.js.map