"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.initRepo = initRepo;
const tslib_1 = require("tslib");
const global_1 = require("../../../config/global");
const secrets_1 = require("../../../config/secrets");
const logger_1 = require("../../../logger");
const remap_1 = require("../../../logger/remap");
const platform_1 = require("../../../modules/platform");
const memCache = tslib_1.__importStar(require("../../../util/cache/memory"));
const clone_1 = require("../../../util/clone");
const git_1 = require("../../../util/git");
const host_rules_1 = require("../../../util/host-rules");
const configured_1 = require("../configured");
const package_files_1 = require("../package-files");
const apis_1 = require("./apis");
const cache_1 = require("./cache");
const config_1 = require("./config");
const vulnerability_1 = require("./vulnerability");
function initializeConfig(config) {
    return {
        ...(0, clone_1.clone)(config),
        errors: [],
        warnings: [],
        branchList: [],
    };
}
function warnOnUnsupportedOptions(config) {
    if (config.filterUnavailableUsers && !platform_1.platform.filterUnavailableUsers) {
        // TODO: types (#22198)
        const platform = global_1.GlobalConfig.get('platform');
        logger_1.logger.warn({ platform }, `Configuration option 'filterUnavailableUsers' is not supported on the current platform.`);
    }
    if (config.expandCodeOwnersGroups && !platform_1.platform.expandGroupMembers) {
        // TODO: types (#22198)
        const platform = global_1.GlobalConfig.get('platform');
        logger_1.logger.warn({ platform }, `Configuration option 'expandCodeOwnersGroups' is not supported on the current platform.`);
    }
}
async function initRepo(config_) {
    package_files_1.PackageFiles.clear();
    let config = initializeConfig(config_);
    await (0, cache_1.resetCaches)();
    logger_1.logger.once.reset();
    memCache.init();
    config = await (0, apis_1.initApis)(config);
    await (0, cache_1.initializeCaches)(config);
    config = await (0, config_1.getRepoConfig)(config);
    (0, remap_1.setRepositoryLogLevelRemaps)(config.logLevelRemap);
    if (config.mode === 'silent') {
        logger_1.logger.info('Repository is running with mode=silent and will not make Issues or PRs by default');
    }
    (0, configured_1.checkIfConfigured)(config);
    warnOnUnsupportedOptions(config);
    config = (0, secrets_1.applySecretsAndVariablesToConfig)({
        config,
    });
    (0, git_1.setUserRepoConfig)(config);
    config = await (0, vulnerability_1.detectVulnerabilityAlerts)(config);
    // istanbul ignore if
    if (config.printConfig) {
        logger_1.logger.info({ config, hostRules: (0, host_rules_1.getAll)() }, 'Full resolved config and hostRules including presets');
    }
    await (0, git_1.cloneSubmodules)(!!config.cloneSubmodules, config.cloneSubmodulesFilter);
    return config;
}
//# sourceMappingURL=index.js.map