"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkConfigMigrationBranch = checkConfigMigrationBranch;
exports.migrationPrExists = migrationPrExists;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const global_1 = require("../../../../config/global");
const logger_1 = require("../../../../logger");
const platform_1 = require("../../../../modules/platform");
const scm_1 = require("../../../../modules/platform/scm");
const common_1 = require("../common");
const commit_message_1 = require("./commit-message");
const create_1 = require("./create");
const rebase_1 = require("./rebase");
async function checkConfigMigrationBranch(config, migratedConfigData) {
    logger_1.logger.debug('checkConfigMigrationBranch()');
    const configMigrationCheckboxState = config.dependencyDashboardChecks?.configMigrationCheckboxState;
    if (!config.configMigration) {
        if (is_1.default.undefined(configMigrationCheckboxState) ||
            configMigrationCheckboxState === 'no-checkbox' ||
            configMigrationCheckboxState === 'unchecked') {
            logger_1.logger.debug('Config migration needed but config migration is disabled and checkbox not checked or not present.');
            return { result: 'no-migration-branch' };
        }
    }
    const configMigrationBranch = (0, common_1.getMigrationBranchName)(config);
    const branchPr = await migrationPrExists(configMigrationBranch, config.baseBranch); // handles open/autoclosed PRs
    if (!branchPr) {
        const commitMessageFactory = new commit_message_1.ConfigMigrationCommitMessageFactory(config, migratedConfigData.filename);
        const prTitle = commitMessageFactory.getPrTitle();
        const closedPrConfig = {
            branchName: configMigrationBranch,
            prTitle,
            state: 'closed',
            targetBranch: config.baseBranch,
        };
        // handles closed PR
        const closedPr = await platform_1.platform.findPr(closedPrConfig);
        // found closed migration PR
        if (closedPr) {
            logger_1.logger.debug('Closed config migration PR found.');
            // if a closed pr exists and the checkbox for config migration is not checked
            // return no-migration-branch result so that the checkbox gets added again
            // we only want to create a config migration pr if the checkbox is checked
            if (configMigrationCheckboxState !== 'checked') {
                logger_1.logger.debug('Config migration is enabled and needed. But a closed pr exists and checkbox is not checked. Skipping migration branch creation.');
                return { result: 'no-migration-branch' };
            }
            logger_1.logger.debug('Closed migration PR found and checkbox is checked. Try to delete this old branch and create a new one.');
            await handlePr(config, closedPr);
        }
    }
    let result;
    if (branchPr) {
        logger_1.logger.debug('Config Migration PR already exists');
        if (await isMigrationBranchModified(config, configMigrationBranch)) {
            logger_1.logger.debug('Config Migration branch has been modified. Skipping branch rebase.');
            result = 'migration-branch-modified';
        }
        else {
            await (0, rebase_1.rebaseMigrationBranch)(config, migratedConfigData);
            if (platform_1.platform.refreshPr) {
                const configMigrationPr = await platform_1.platform.getBranchPr(configMigrationBranch, config.baseBranch);
                if (configMigrationPr) {
                    await platform_1.platform.refreshPr(configMigrationPr.number);
                }
            }
            result = 'migration-branch-exists';
        }
    }
    else {
        logger_1.logger.debug('Config Migration PR does not exist');
        logger_1.logger.debug('Need to create migration PR');
        await (0, create_1.createConfigMigrationBranch)(config, migratedConfigData);
        result = 'migration-branch-exists';
    }
    if (!global_1.GlobalConfig.get('dryRun')) {
        await scm_1.scm.checkoutBranch(configMigrationBranch);
    }
    return {
        migrationBranch: configMigrationBranch,
        result,
    };
}
async function migrationPrExists(branchName, targetBranch) {
    return await platform_1.platform.getBranchPr(branchName, targetBranch);
}
async function handlePr(config, pr) {
    if (await scm_1.scm.branchExists(pr.sourceBranch)) {
        if (global_1.GlobalConfig.get('dryRun')) {
            logger_1.logger.info('DRY-RUN: Would delete branch ' + pr.sourceBranch);
        }
        else {
            await scm_1.scm.deleteBranch(pr.sourceBranch);
        }
    }
}
async function isMigrationBranchModified(config, configMigrationBranch) {
    const baseBranch = config.defaultBranch;
    return await scm_1.scm.isBranchModified(configMigrationBranch, baseBranch);
}
//# sourceMappingURL=index.js.map