"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setBranchCache = setBranchCache;
const error_messages_1 = require("../../constants/error-messages");
const logger_1 = require("../../logger");
const platform_1 = require("../../modules/platform");
const scm_1 = require("../../modules/platform/scm");
const repository_1 = require("../../util/cache/repository");
const behind_base_branch_cache_1 = require("../../util/git/behind-base-branch-cache");
const conflicts_cache_1 = require("../../util/git/conflicts-cache");
const modified_cache_1 = require("../../util/git/modified-cache");
const pristine_1 = require("../../util/git/pristine");
const pr_cache_1 = require("./update/pr/pr-cache");
function generateBranchUpgradeCache(upgrade) {
    const { datasource, depName, depType, displayPending, packageName, fixedVersion, currentVersion, newVersion, currentValue, newValue, currentDigest, newDigest, packageFile, sourceUrl, remediationNotPossible, updateType, } = upgrade;
    const result = {
        datasource,
        depName,
        depType,
        displayPending,
        fixedVersion,
        currentVersion,
        currentValue,
        newValue,
        newVersion,
        currentDigest,
        newDigest,
        packageFile,
        sourceUrl,
        remediationNotPossible,
        updateType,
    };
    if (packageName) {
        result.packageName = packageName;
    }
    return result;
}
async function generateBranchCache(branch) {
    const { baseBranch, branchName, prBlockedBy, prTitle, result } = branch;
    try {
        const branchSha = await scm_1.scm.getBranchCommit(branchName);
        const baseBranchSha = await scm_1.scm.getBranchCommit(baseBranch);
        const pristine = (0, pristine_1.getCachedPristineResult)(branchName);
        let prNo = null;
        let isModified;
        let isBehindBase;
        let isConflicted;
        if (baseBranchSha && branchSha) {
            const branchPr = await platform_1.platform.getBranchPr(branchName, baseBranch);
            if (branchPr) {
                prNo = branchPr.number;
            }
            isModified = (0, modified_cache_1.getCachedModifiedResult)(branchName, branchSha) ?? undefined;
            isBehindBase =
                (0, behind_base_branch_cache_1.getCachedBehindBaseResult)(branchName, branchSha, baseBranch, baseBranchSha) ?? undefined;
            isConflicted =
                (0, conflicts_cache_1.getCachedConflictResult)(branchName, branchSha, baseBranch, baseBranchSha) ?? undefined;
        }
        else if (baseBranchSha && !branchSha && branch.prNo) {
            // if branch was deleted/ PR exists and ignored
            prNo = branch.prNo;
        }
        const automerge = !!branch.automerge;
        const upgrades = branch.upgrades
            ? branch.upgrades.map(generateBranchUpgradeCache)
            : [];
        const commitFingerprint = branch.commitFingerprint;
        const prCache = (0, pr_cache_1.getPrCache)(branchName);
        return {
            automerge,
            baseBranchSha,
            baseBranch,
            commitFingerprint,
            branchName,
            isBehindBase,
            isConflicted,
            isModified,
            prBlockedBy,
            pristine,
            prCache,
            prNo,
            prTitle,
            result,
            sha: branchSha,
            upgrades,
        };
    }
    catch (error) {
        const err = error.err ?? error; // external host error nests err
        const errCodes = [401, 404];
        // istanbul ignore if
        if (errCodes.includes(err.response?.statusCode)) {
            logger_1.logger.warn({ err, branchName }, 'HTTP error generating branch cache');
            return null;
        }
        if (err.message === error_messages_1.REPOSITORY_CHANGED) {
            throw err;
        }
        logger_1.logger.error({ err, branchName }, 'Error generating branch cache');
        return null;
    }
}
async function setBranchCache(branches) {
    const branchCaches = [];
    for (const branch of branches) {
        const branchCache = await generateBranchCache(branch);
        if (branchCache) {
            branchCaches.push(branchCache);
        }
    }
    (0, repository_1.getCache)().branches = branchCaches;
}
//# sourceMappingURL=cache.js.map