"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.hostRulesFromEnv = hostRulesFromEnv;
const logger_1 = require("../../../../logger");
const datasource_1 = require("../../../../modules/datasource");
function isAuthField(x) {
    return x === 'token' || x === 'username' || x === 'password';
}
function isHttpsAuthField(x) {
    return (x === 'httpscertificate' ||
        x === 'httpsprivatekey' ||
        x === 'httpscertificateauthority');
}
function restoreHttpsAuthField(x) {
    switch (x) {
        case 'httpsprivatekey':
            return 'httpsPrivateKey';
        case 'httpscertificate':
            return 'httpsCertificate';
        case 'httpscertificateauthority':
            return 'httpsCertificateAuthority';
    }
    return x;
}
function setHostRuleValue(rule, key, value) {
    if (value !== undefined) {
        switch (key) {
            case 'token':
            case 'username':
            case 'password':
            case 'httpsCertificateAuthority':
            case 'httpsCertificate':
            case 'httpsPrivateKey':
                rule[key] = value;
        }
    }
}
function hostRulesFromEnv(env) {
    const datasources = new Set((0, datasource_1.getDatasourceList)());
    const platforms = new Set(['github']);
    const hostRules = [];
    const npmEnvPrefixes = ['npm_config_', 'npm_lifecycle_', 'npm_package_'];
    for (const envName of Object.keys(env).sort()) {
        if (['GITHUB_COM_TOKEN', 'RENOVATE_GITHUB_COM_TOKEN'].includes(envName)) {
            continue;
        }
        if (npmEnvPrefixes.some((prefix) => envName.startsWith(prefix))) {
            logger_1.logger.trace('Ignoring npm env: ' + envName);
            continue;
        }
        // Double underscore __ is used in place of hyphen -
        const splitEnv = envName
            .replace(/^RENOVATE_/, '')
            .toLowerCase()
            .replace(/__/g, '-')
            .split('_');
        const hostType = splitEnv.shift();
        if (datasources.has(hostType) ||
            (platforms.has(hostType) && splitEnv.length > 1)) {
            let suffix = splitEnv.pop();
            if (isAuthField(suffix) || isHttpsAuthField(suffix)) {
                suffix = restoreHttpsAuthField(suffix);
                let matchHost = undefined;
                const rule = {};
                setHostRuleValue(rule, suffix, env[envName]);
                if (splitEnv.length === 0) {
                    // host-less rule
                }
                else if (splitEnv.length === 1) {
                    logger_1.logger.warn({ env: envName }, 'Cannot parse env');
                    continue;
                }
                else {
                    matchHost = splitEnv.join('.');
                }
                const existingRule = hostRules.find((hr) => hr.hostType === hostType && hr.matchHost === matchHost);
                logger_1.logger.debug(`Converting ${envName} into a global host rule`);
                if (existingRule) {
                    // Add current field to existing rule
                    setHostRuleValue(existingRule, suffix, env[envName]);
                }
                else {
                    // Create a new rule
                    const newRule = {
                        hostType,
                    };
                    if (matchHost) {
                        newRule.matchHost = matchHost;
                    }
                    setHostRuleValue(newRule, suffix, env[envName]);
                    hostRules.push(newRule);
                }
            }
        }
    }
    return hostRules;
}
//# sourceMappingURL=host-rules-from-env.js.map